<?php

namespace Drupal\laposta_webform\Plugin\WebformElement;

use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Plugin\WebformElement\OptionsBase;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformSubmissionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'laposta_list_checkboxes' element.
 *
 * @WebformElement(
 *   id = "laposta_list_checkboxes",
 *   label = @Translation("Laposta list checkboxes"),
 *   description = @Translation("Provides checkboxes populated with Laposta mailing lists."),
 *   category = @Translation("Options elements"),
 * )
 */
class LapostaListCheckboxes extends OptionsBase {

  /**
   * The Laposta API service.
   *
   * @var \Drupal\laposta_webform\Service\LapostaApi
   */
  protected $lapostaApi;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->lapostaApi = $container->get('laposta_webform.api');
    return $instance;
  }

  /**
   * Gets the Laposta API service.
   *
   * @return \Drupal\laposta_webform\Service\LapostaApi
   *   The Laposta API service.
   */
  protected function getLapostaApi() {
    if (!$this->lapostaApi) {
      $this->lapostaApi = \Drupal::service('laposta_webform.api');
    }
    return $this->lapostaApi;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineDefaultProperties() {
    $properties = parent::defineDefaultProperties();
    $properties['options_display'] = 'one_column';
    $properties['laposta_filter_lists'] = '';
    // Remove 'options' from parent - we populate dynamically.
    unset($properties['options']);
    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    // Hide the options element since options come from Laposta.
    if (isset($form['options'])) {
      $form['options']['#access'] = FALSE;
    }

    // Get element properties.
    $element_properties = $form_state->get('element_properties') ?? [];

    // Get available lists.
    $lists = $this->getLapostaApi()->getLists();

    // Add Laposta settings in the element details section.
    $form['element']['laposta_filter_lists'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Show only these lists'),
      '#description' => $this->t('Select which lists users can subscribe to. Leave empty to show all lists.'),
      '#options' => $lists,
      '#weight' => 10,
    ];

    if (!empty($lists)) {
      // Get currently selected lists (may be comma-separated string or array).
      $stored_value = $element_properties['laposta_filter_lists'] ?? '';
      if (is_array($stored_value)) {
        $selected_ids = $stored_value;
      }
      elseif (!empty($stored_value) && is_string($stored_value)) {
        $selected_ids = explode(',', $stored_value);
      }
      else {
        $selected_ids = [];
      }
      $form['element']['laposta_filter_lists']['#default_value'] = $selected_ids;
    }
    else {
      $form['element']['laposta_filter_lists']['#access'] = FALSE;
      $form['element']['laposta_status'] = [
        '#type' => 'item',
        '#title' => $this->t('Laposta Status'),
        '#markup' => '<span style="color: red;">' . $this->t('No lists available. Please check your API key configuration.') . '</span>',
        '#weight' => 10,
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);

    // Convert multi-select array to comma-separated string.
    $selected = $form_state->getValue(['properties', 'laposta_filter_lists']) ?? [];
    if (is_array($selected)) {
      $selected = array_values(array_filter($selected));
      $form_state->setValue(['properties', 'laposta_filter_lists'], implode(',', $selected));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function prepare(array &$element, WebformSubmissionInterface $webform_submission = NULL) {
    // Set the element type to checkboxes.
    $element['#type'] = 'checkboxes';

    // Get lists from Laposta API.
    $lists = $this->getLapostaApi()->getLists();

    // Filter by allowed lists if configured.
    $filter_value = $element['#laposta_filter_lists'] ?? '';
    if (!empty($filter_value)) {
      if (is_array($filter_value)) {
        // Multi-select returns ['id' => 'id', ...], get just the values.
        $allowed_ids = array_values(array_filter($filter_value));
      }
      elseif (is_string($filter_value)) {
        $allowed_ids = explode(',', $filter_value);
      }
      else {
        $allowed_ids = [];
      }
      if (!empty($allowed_ids)) {
        $allowed_keys = array_flip($allowed_ids);
        $lists = array_intersect_key($lists, $allowed_keys);
      }
    }

    // Set options before calling parent.
    $element['#options'] = $lists;

    parent::prepare($element, $webform_submission);

    // Set options again after parent in case it was overwritten.
    $element['#options'] = $lists;
  }

  /**
   * {@inheritdoc}
   */
  public function getTestValues(array $element, WebformInterface $webform, array $options = []) {
    $lists = $this->getLapostaApi()->getLists();
    if (empty($lists)) {
      return [];
    }

    // Return first two list IDs as test values.
    if (count($lists) > 1) {
      return [array_slice(array_keys($lists), 0, 2)];
    }

    return [array_keys($lists)];
  }

  /**
   * {@inheritdoc}
   */
  public function hasMultipleValues(array $element) {
    // Checkboxes always have multiple values.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function isMultiline(array $element) {
    return TRUE;
  }

}
