<?php

namespace Drupal\laposta_webform\Plugin\WebformElement;

use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Plugin\WebformElement\OptionsBase;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformSubmissionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'laposta_list_select' element.
 *
 * @WebformElement(
 *   id = "laposta_list_select",
 *   label = @Translation("Laposta list select"),
 *   description = @Translation("Provides a select element populated with Laposta mailing lists."),
 *   category = @Translation("Options elements"),
 * )
 */
class LapostaListSelect extends OptionsBase {

  /**
   * The Laposta API service.
   *
   * @var \Drupal\laposta_webform\Service\LapostaApi
   */
  protected $lapostaApi;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->lapostaApi = $container->get('laposta_webform.api');
    return $instance;
  }

  /**
   * Gets the Laposta API service.
   *
   * @return \Drupal\laposta_webform\Service\LapostaApi
   *   The Laposta API service.
   */
  protected function getLapostaApi() {
    if (!$this->lapostaApi) {
      $this->lapostaApi = \Drupal::service('laposta_webform.api');
    }
    return $this->lapostaApi;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineDefaultProperties() {
    $properties = parent::defineDefaultProperties();
    $properties['multiple'] = FALSE;
    $properties['laposta_empty_option'] = '';
    $properties['laposta_filter_lists'] = '';
    // Remove 'options' from parent - we populate dynamically.
    unset($properties['options']);
    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    // Hide the options element since options come from Laposta.
    if (isset($form['options'])) {
      $form['options']['#access'] = FALSE;
    }

    // Get element properties.
    $element_properties = $form_state->get('element_properties') ?? [];

    // Get available lists.
    $lists = $this->getLapostaApi()->getLists();

    // Add Laposta settings in the element details section.
    $form['element']['laposta_filter_lists'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Show only these lists'),
      '#description' => $this->t('Select which lists users can subscribe to. Leave empty to show all lists.'),
      '#options' => $lists,
      '#weight' => 10,
    ];

    $form['element']['multiple'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow multiple selections'),
      '#description' => $this->t('Allow visitors to select multiple lists.'),
      '#default_value' => $element_properties['multiple'] ?? FALSE,
      '#weight' => 11,
    ];

    $form['element']['laposta_empty_option'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Empty option label'),
      '#description' => $this->t('The label for the empty option (e.g., "- Select a list -"). Leave empty to not show an empty option. Only applies when multiple selections is disabled.'),
      '#default_value' => $element_properties['laposta_empty_option'] ?? '',
      '#weight' => 12,
    ];

    if (!empty($lists)) {
      // Get currently selected lists (may be comma-separated string or array).
      $stored_value = $element_properties['laposta_filter_lists'] ?? '';
      if (is_array($stored_value)) {
        $selected_ids = $stored_value;
      }
      elseif (!empty($stored_value) && is_string($stored_value)) {
        $selected_ids = explode(',', $stored_value);
      }
      else {
        $selected_ids = [];
      }
      $form['element']['laposta_filter_lists']['#default_value'] = $selected_ids;
    }
    else {
      $form['element']['laposta_filter_lists']['#access'] = FALSE;
      $form['element']['laposta_empty_option']['#access'] = FALSE;
      $form['element']['laposta_status'] = [
        '#type' => 'item',
        '#title' => $this->t('Laposta Status'),
        '#markup' => '<span style="color: red;">' . $this->t('No lists available. Please check your API key configuration.') . '</span>',
        '#weight' => 10,
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);

    // Convert multi-select array to comma-separated string.
    $selected = $form_state->getValue(['properties', 'laposta_filter_lists']) ?? [];
    if (is_array($selected)) {
      $selected = array_values(array_filter($selected));
      $form_state->setValue(['properties', 'laposta_filter_lists'], implode(',', $selected));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function prepare(array &$element, WebformSubmissionInterface $webform_submission = NULL) {
    // Set the element type to select.
    $element['#type'] = 'select';

    // Set multiple if configured.
    if (!empty($element['#multiple'])) {
      $element['#multiple'] = TRUE;
    }

    // Get lists from Laposta API.
    $lists = $this->getLapostaApi()->getLists();

    // Filter by allowed lists if configured.
    $filter_value = $element['#laposta_filter_lists'] ?? '';
    if (!empty($filter_value)) {
      if (is_array($filter_value)) {
        // Multi-select returns ['id' => 'id', ...], get just the values.
        $allowed_ids = array_values(array_filter($filter_value));
      }
      elseif (is_string($filter_value)) {
        $allowed_ids = explode(',', $filter_value);
      }
      else {
        $allowed_ids = [];
      }
      if (!empty($allowed_ids)) {
        $allowed_keys = array_flip($allowed_ids);
        $lists = array_intersect_key($lists, $allowed_keys);
      }
    }

    // Build options.
    $options = [];

    // Only add empty option for single select (not multiple).
    $is_multiple = !empty($element['#multiple']);
    $empty_option = $element['#laposta_empty_option'] ?? '';
    if (!empty($empty_option) && !$is_multiple) {
      $options[''] = $empty_option;
    }

    $options += $lists;

    // Set options before calling parent.
    $element['#options'] = $options;

    parent::prepare($element, $webform_submission);

    // Set options again after parent in case it was overwritten.
    $element['#options'] = $options;
  }

  /**
   * {@inheritdoc}
   */
  public function getTestValues(array $element, WebformInterface $webform, array $options = []) {
    $lists = $this->getLapostaApi()->getLists();
    if (empty($lists)) {
      return [];
    }

    // Return multiple list IDs if multiple selection is enabled.
    $is_multiple = !empty($element['#multiple']);
    if ($is_multiple && count($lists) > 1) {
      return [array_slice(array_keys($lists), 0, 2)];
    }

    // Return the first list ID as test value.
    return [array_key_first($lists)];
  }

}
