<?php

declare(strict_types=1);

namespace Drupal\lawwwing\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\RoleInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Provides the settings form for the Lawwwing module.
 */
class LawwwingSettingsForm extends ConfigFormBase {

  use AutowireTrait;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'lawwwing_settings_form';
  }

  public function __construct(
    ConfigFactoryInterface $config_factory,
    #[Autowire(service: 'config.typed')]
    $typedConfigManager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['lawwwing.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    // Script ID field.
    $form['script_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Lawwwing ID'),
      '#description' => $this->t("Enter your Lawwwing ID. The script tag will be automatically inserted into the 'head' section of your website."),
      '#config_target' => 'lawwwing.settings:script_id',
      '#required' => TRUE,
      '#attributes' => [
        'placeholder' => 'xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx',
      ],
    ];

    // Add in admin theme section checkbox.
    $form['active_in_admin'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Add in the admin pages'),
      '#description' => $this->t('Check to add the script in the admin pages of the site.'),
      '#config_target' => 'lawwwing.settings:active_in_admin',
    ];

    // Allowed roles.
    $form['allowed_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed roles'),
      '#options' => array_map(
        fn (RoleInterface $role): string => $role->label(),
        $this->entityTypeManager->getStorage('user_role')->loadMultiple(),
      ),
      '#description' => $this->t('Select the user roles for which the script will be displayed.'),
      '#config_target' => 'lawwwing.settings:allowed_roles',
    ];

    return parent::buildForm($form, $form_state);
  }

}
