<?php

declare(strict_types=1);

namespace Drupal\lawwwing\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Hooks for Lawwwing module.
 */
final class Hooks {

  use AutowireTrait;

  public function __construct(
    private AdminContext $adminContext,
    private ConfigFactoryInterface $configFactory,
    private AccountProxyInterface $currentUser,
  ) {
  }

  /**
   * Attaches the Lawwwing script to the page.
   *
   * @param array $attachments
   *   The attachments array to which the script will be added.
   */
  #[Hook('page_attachments')]
  public function insertScript(array &$attachments): void {
    $config = $this->configFactory->get('lawwwing.settings');
    $script_id = $config->get('script_id');

    $attachments['#cache']['tags'][] = 'config:lawwwing.settings';

    // Do nothing if no script ID is set.
    if (!$script_id) {
      return;
    }

    // Check if the current route is and admin one and if the script should be
    // added there.
    $is_admin = $this->adminContext->isAdminRoute();
    if ($is_admin && !$config->get('active_in_admin')) {
      return;
    }

    // Filter by roles.
    $allowed_roles = $config->get('allowed_roles') ?: [];
    if (empty($allowed_roles) || !array_intersect($allowed_roles, $this->currentUser->getRoles())) {
      return;
    }

    $attachments['#attached']['html_head'][] = [
      [
        '#type' => 'html_tag',
        '#tag' => 'script',
        '#value' => '',
        '#attributes' => [
          'src' => "https://cdn.lawwwing.com/widgets/current/{$script_id}/cookie-widget.min.js",
          'type' => 'text/javascript',
          'data-lwid' => $script_id,
        ],
      ],
      'lawwing',
    ];
  }

}
