<?php

namespace Drupal\layout_builder_ajax_blocks\Controller;

use Drupal\block\BlockInterface;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Block\BlockManagerInterface;
use Drupal\Core\Block\BlockPluginInterface;
use Drupal\Core\Block\TitleBlockPluginInterface;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Controller\TitleResolverInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Plugin\Context\ContextHandlerInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Utility\Token;
use Drupal\node\NodeInterface;
use Drupal\taxonomy\TermInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Block loader controller class.
 */
class BlockLoader extends ControllerBase {

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Plugin\Context\ContextHandlerInterface $contextHandler
   *   The context handler.
   * @param \Drupal\Core\Plugin\Context\ContextRepositoryInterface $contextRepository
   *   The context repository.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user account.
   * @param \Drupal\Core\Block\BlockManagerInterface $blockManager
   *   The block manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The route match service.
   * @param \Drupal\Core\Controller\TitleResolverInterface $titleResolver
   *   The title resolver.
   * @param \Drupal\Core\Utility\Token $token
   *   The token utility service.
   */
  public function __construct(
    #[Autowire(service: 'context.handler')] protected ContextHandlerInterface $contextHandler,
    #[Autowire(service: 'context.repository')] protected ContextRepositoryInterface $contextRepository,
    #[Autowire(service: 'current_user')] protected AccountInterface $account,
    #[Autowire(service: 'plugin.manager.block')] protected BlockManagerInterface $blockManager,
    #[Autowire(service: 'renderer')] protected RendererInterface $renderer,
    #[Autowire(service: 'request_stack')] protected RequestStack $requestStack,
    #[Autowire(service: 'current_route_match')] protected RouteMatchInterface $routeMatch,
    #[Autowire(service: 'title_resolver')] protected TitleResolverInterface $titleResolver,
    #[Autowire(service: 'token')] protected Token $token,
  ) {
  }

  /**
   * Load: no context.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $block_id
   *   The block id.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Returns the response.
   */
  public function load(Request $request, string $block_id): JsonResponse {
    return $this->loadContext($request, $block_id);
  }

  /**
   * Load: node context.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $block_id
   *   The block id.
   * @param \Drupal\node\NodeInterface $node
   *   The node.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Returns the response.
   */
  public function loadNodeContext(Request $request, string $block_id, NodeInterface $node): JsonResponse {
    return $this->loadContext($request, $block_id, 'node', $node);
  }

  /**
   * Load: term context.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $block_id
   *   The block id.
   * @param \Drupal\taxonomy\TermInterface $term
   *   The taxonomy term.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Returns the response.
   */
  public function loadTaxonomyTermContext(Request $request, string $block_id, TermInterface $term): JsonResponse {
    return $this->loadContext($request, $block_id, 'term', $term);
  }

  /**
   * Load: user context.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $block_id
   *   The block id.
   * @param \Drupal\user\UserInterface $user
   *   The user.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Returns the response.
   */
  public function loadUserContext(Request $request, string $block_id, UserInterface $user): JsonResponse {
    return $this->loadContext($request, $block_id, 'user', $user);
  }

  /**
   * Builds the render array for a block.
   *
   * @param string $id
   *   The string of block plugin to render.
   * @param array $configuration
   *   (optional) Pass on any configuration to the plugin block.
   *
   * @return array
   *   A renderable array representing the content of the block.
   */
  protected function build(string $id, array $configuration): array {
    $configuration += [
      'label_display' => BlockPluginInterface::BLOCK_LABEL_VISIBLE,
    ];

    /** @var \Drupal\Core\Block\BlockPluginInterface $block_plugin */
    $block_plugin = $this->blockManager->createInstance($id, $configuration);

    $build = [];
    $access = $block_plugin->access($this->account, TRUE);
    if ($access->isAllowed()) {

      // Title block needs a special treatment.
      if ($block_plugin instanceof TitleBlockPluginInterface) {
        $request = $this->requestStack->getCurrentRequest();
        $title = $this->titleResolver->getTitle($request, $this->routeMatch->getRouteObject());
        $block_plugin->setTitle($title);
      }

      // Place the content returned by the block plugin into a 'content' child
      // element, as a way to allow the plugin to have complete control of its
      // properties and rendering (for instance, its own #theme) without
      // conflicting with the properties used above.
      $build['content'] = $block_plugin->build();
      if ($block_plugin instanceof TitleBlockPluginInterface) {
        $build['content']['#cache']['contexts'][] = 'url';
      }

      // Some blocks return null instead of array when empty.
      // @see https://www.drupal.org/project/drupal/issues/3212354
      if (is_array($build['content']) && !Element::isEmpty($build['content'])) {
        $build += [
          '#theme' => 'block',
          '#id' => $configuration['id'] ?? NULL,
          '#contextual_links' => [],
          '#configuration' => $block_plugin->getConfiguration(),
          '#plugin_id' => $block_plugin->getPluginId(),
          '#base_plugin_id' => $block_plugin->getBaseId(),
          '#derivative_plugin_id' => $block_plugin->getDerivativeId(),
        ];
        // Semantically, the content returned by the plugin is the block, and in
        // particular, #attributes and #contextual_links is information about
        // the *entire* block. Therefore, we must move these properties into the
        // top-level element.
        foreach (['#attributes', '#contextual_links'] as $property) {
          if (isset($build['content'][$property])) {
            $build[$property] = $build['content'][$property];
            unset($build['content'][$property]);
          }
        }
      }
    }

    CacheableMetadata::createFromRenderArray($build)
      ->addCacheableDependency($access)
      ->addCacheableDependency($block_plugin)
      ->applyTo($build);

    if (!isset($build['#cache']['keys'])) {
      $build['#cache']['keys'] = [
        'layout_builder_ajax_blocks',
        $id,
        '[configuration]=' . hash('sha256', serialize($configuration)),
      ];
    }

    return $build;
  }

  /**
   * Get filtered block configuration data.
   *
   * @param array $data
   *   The configuration data.
   *
   * @return array
   *   The filtered configuration data.
   */
  protected function filterConfiguration(array $data): array {
    foreach ($data as $key => $value) {
      if (is_array($value)) {
        $this->filterConfiguration($value);
      }
      else {
        $data[$key] = Xss::filter($value);
      }
    }
    return $data;
  }

  /**
   * Get block configuration.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return array
   *   The merged and sanitized configuration.
   */
  protected function getBlockConfiguration(Request $request): array {
    $body = json_decode($request->getContent(), TRUE);
    $configuration = $body['config'] ?? [];

    return $this->filterConfiguration($configuration);
  }

  /**
   * Get the block instance.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $block_id
   *   The block/plugin id.
   * @param array $configuration
   *   The block configuration.
   *
   * @return array|null
   *   Returns the block render array/instance.
   */
  protected function getBlockInstance(Request $request, string $block_id, array $configuration): ?array {
    $block = $this->entityTypeManager()->getStorage('block')
      ->load($block_id);
    if ($block instanceof BlockInterface) {
      $block_instance = $this->getBlockInstanceBlock($block, $request);
    }
    else {
      $block_instance = $this->getBlockInstanceBlockPlugin($block_id, $configuration);
    }

    return $block_instance;
  }

  /**
   * Get the block instance (block variant).
   *
   * @param \Drupal\Core\Block\BlockInterface $block
   *   The block.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return array|null
   *   Returns the block render array/instance.
   */
  protected function getBlockInstanceBlock(BlockInterface $block, Request $request): ?array {
    $block_instance = NULL;

    $plugin_id = $request->get('plugin_id', '');
    if (!empty($plugin_id)) {

      // Check if block has special plugin and add it to dependency.
      $plugin = $block->getPlugin();
      if (is_object($plugin) && $plugin->getPluginId() == $plugin_id) {
        $configuration = $plugin->getConfiguration();
      }
      $block_instance = $this->build($plugin_id, $configuration);
    }

    return $block_instance;
  }

  /**
   * Get the block instance (block plugin variant).
   *
   * @param string $block_id
   *   The block plugin id.
   * @param array $configuration
   *   The block configuration.
   *
   * @return array|null
   *   Returns the block render array/instance.
   */
  protected function getBlockInstanceBlockPlugin(string $block_id, array $configuration): ?array {
    $block_instance = NULL;

    $block_plugin = $this->blockManager->createInstance($block_id, $configuration);
    if (
      $block_plugin instanceof BlockPluginInterface &&
      $block_plugin->getPluginId() != 'broken' &&
      $block_plugin->access($this->account, TRUE)->isAllowed()
    ) {
      $block_instance = $this->build($block_id, $configuration);
    }

    return $block_instance;
  }

  /**
   * Load context.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $block_id
   *   The block id.
   * @param string $context_type
   *   The type of context entity - node, term or user.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The context entity (optional).
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Returns the ajax block response.
   */
  protected function loadContext(Request $request, string $block_id, ?string $context_type = NULL, ?ContentEntityInterface $entity = NULL): JsonResponse {
    $response = new JsonResponse();

    if (!empty($block_id)) {
      $configuration = $this->getBlockConfiguration($request);
      $block_instance = $this->getBlockInstance($request, $block_id, $configuration);
      if (!empty($block_instance)) {
        $rendered_block = $this->renderer->renderRoot($block_instance);

        $context = [];
        $url = '/layout_builder_ajax_blocks/load/' . $block_id;
        if (!empty($context_type) && !empty($entity)) {
          $context = [
            $context_type => $context,
          ];
          $url = '/layout_builder_ajax_blocks/load/' . $block_id . '/' . $context_type . '/' . $entity->id();
        }

        // Token replacement and ajax link cleanup.
        $rendered_block = $this->token->replace($rendered_block, $context);
        $rendered_block = str_replace($url, '', trim($rendered_block));

        // Don not cache ajax blocks.
        $max_age = 0;
        $response->setMaxAge($max_age);
        $response->setSharedMaxAge($max_age);
        $response->setData(
          [
            'content' => $rendered_block,
          ]
        );
      }
    }

    return $response;
  }

}
