<?php

namespace Drupal\layout_builder_ajax_blocks\EventSubscriber;

use Drupal\block_content\Access\RefinableDependentAccessInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\BlockPluginInterface;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Render\PreviewFallbackInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\layout_builder\Access\LayoutPreviewAccessAllowed;
use Drupal\layout_builder\Event\SectionComponentBuildRenderArrayEvent;
use Drupal\layout_builder\LayoutBuilderEvents;
use Drupal\layout_builder\Plugin\Block\InlineBlock;
use Drupal\node\NodeInterface;
use Drupal\taxonomy\TermInterface;
use Drupal\views\Plugin\Block\ViewsBlock;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber to the initial render array.
 */
class BlockComponentRenderArraySubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The default cache backend.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The route match service.
   */
  public function __construct(
    protected AccountInterface $currentUser,
    protected CacheBackendInterface $cache,
    protected RouteMatchInterface $routeMatch,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[LayoutBuilderEvents::SECTION_COMPONENT_BUILD_RENDER_ARRAY] = ['onBuildRender', 100];
    return $events;
  }

  /**
   * Add each component's block styles to the render array.
   *
   * @param \Drupal\layout_builder\Event\SectionComponentBuildRenderArrayEvent $event
   *   The section component render event.
   */
  public function onBuildRender(SectionComponentBuildRenderArrayEvent $event) {
    $block = $event->getPlugin();
    if (!$block instanceof BlockPluginInterface) {
      return;
    }

    // Set block access dependency even if we are not checking access on
    // this level. The block itself may render another
    // RefinableDependentAccessInterface object and need to pass on this value.
    if ($block instanceof RefinableDependentAccessInterface) {
      $contexts = $event->getContexts();
      if (isset($contexts['layout_builder.entity'])) {
        if ($entity = $contexts['layout_builder.entity']->getContextValue()) {
          if ($event->inPreview()) {
            // If previewing in Layout Builder allow access.
            $block->setAccessDependency(new LayoutPreviewAccessAllowed());
          }
          else {
            $block->setAccessDependency($entity);
          }
        }
      }
    }

    // Only check access if the component is not being previewed.
    if ($event->inPreview()) {
      $access = AccessResult::allowed()->setCacheMaxAge(0);
    }
    else {
      $access = $block->access($this->currentUser, TRUE);
    }

    $event->addCacheableDependency($access);
    if ($access->isAllowed()) {
      $event->addCacheableDependency($block);

      // @todo Revisit after https://www.drupal.org/node/3027653, as this will
      //   provide a better way to remove contextual links from Views blocks.
      //   Currently, doing this requires setting
      //   \Drupal\views\ViewExecutable::$showAdminLinks() to false before the
      //   Views block is built.
      if ($block instanceof ViewsBlock && $event->inPreview()) {
        $block->getViewExecutable()->setShowAdminLinks(FALSE);
      }

      // Ajax block if specified and not managing layout.
      $block_id = $block->getPluginId();
      $block_config = $block->getConfiguration();
      $ajax_enabled = $event->getComponent()->get('layout_builder_ajax_blocks_enabled');
      $initial_load = TRUE;
      $cache_attached = $ajax_enabled;

      // The first time this block is loaded, let it load completely and cache
      // all of the #attached data so it can be included with the ajax blocks
      // placeholder. Use two cache keys to keep track of empty #attached data
      // as well.
      $cid_attached_data = 'layout_builder_ajax_blocks:attached_data:' . $block_id;
      $cid_attached = 'layout_builder_ajax_blocks:attached:' . $block_id;
      $cache_data_attached_data = $this->cache->get($cid_attached_data);
      $cache_data_attached = $this->cache->get($cid_attached);
      if ($cache_data_attached) {
        $initial_load = FALSE;
      }

      if ($ajax_enabled && !$initial_load && !$event->inPreview()) {
        $content = [
          '#theme' => 'layout_builder_ajax_blocks_block',
          '#block_id' => $block_id,
          '#block_settings' => $block_config,
          '#weight' => $event->getComponent()->getWeight(),
        ];
        if (!empty($cache_data_attached_data->data)) {
          $content['#attached'] = $cache_data_attached_data->data;
        }

        $content['#attached']['library'][] = 'layout_builder_ajax_blocks/loader';
        $content['#attached']['drupalSettings']['layout_builder_ajax_blocks']['blocks'][$block_id] = $block_config;

        // Add contextual links.
        if ($this->currentUser->hasPermission('access contextual links') && !empty($build['#contextual_links'])) {
          $content['#layout_builder_ajax_blocks_contextual_links'] = [
            '#type' => 'contextual_links_placeholder',
            '#id' => _contextual_links_to_id($content['#contextual_links']),
          ];
        }

        // Block context.
        if (!empty($block_config['context_mapping'])) {
          if (isset($block_config['context_mapping']['node'])) {
            $content['#attached']['drupalSettings']['layout_builder_ajax_blocks']['blocks'][$block_id]['current_node'] = $this->getCurrentNodeId();
          }
          elseif (isset($block_config['context_mapping']['user'])) {
            $content['#attached']['drupalSettings']['layout_builder_ajax_blocks']['blocks'][$block_id]['current_user'] = $this->currentUser->id();
          }
          elseif (isset($block_config['context_mapping']['taxonomy_term'])) {
            $content['#attached']['drupalSettings']['layout_builder_ajax_blocks']['blocks'][$block_id]['current_term'] = $this->getCurrentTaxonomyTermId();
          }
        }
      }
      else {
        $content = $block->build();

        // On initial load, cache the attached data. Second load will attach
        // stuff to the page and load the block via ajax.
        if ($initial_load && $cache_attached) {
          $this->cache->set($cid_attached, TRUE);
          if (!empty($content['#attached'])) {
            $this->cache->set($cid_attached_data, $content['#attached']);
          }

          $content['#cache']['max-age'] = 0;
        }
      }

      // We don't output the block render data if there are no render elements
      // found, but we want to capture the cache metadata from the block
      // regardless.
      $event->addCacheableDependency(CacheableMetadata::createFromRenderArray($content));

      $is_content_empty = Element::isEmpty($content);
      $is_placeholder_ready = $event->inPreview() && $block instanceof PreviewFallbackInterface;
      // If the content is empty and no placeholder is available, return.
      if ($is_content_empty && !$is_placeholder_ready) {
        return;
      }

      if ($ajax_enabled && !$initial_load && !$event->inPreview()) {
        $build = $content;
      }
      else {
        $build = [
          '#theme' => 'block',
          '#configuration' => $block->getConfiguration(),
          '#plugin_id' => $block->getPluginId(),
          '#base_plugin_id' => $block->getBaseId(),
          '#derivative_plugin_id' => $block->getDerivativeId(),
          '#in_preview' => $event->inPreview(),
          '#weight' => $event->getComponent()->getWeight(),
        ];
      }

      // Place the $content returned by the block plugin into a 'content' child
      // element, as a way to allow the plugin to have complete control of its
      // properties and rendering (for instance, its own #theme) without
      // conflicting with the properties used above, or alternate ones used by
      // alternate block rendering approaches in contributed modules. However,
      // the use of a child element is an implementation detail of this
      // particular block rendering approach. Semantically, the content returned
      // by the block plugin, and in particular, attributes and contextual links
      // are information that belong to the entire block. Therefore, we must
      // move these properties from $content and merge them into the top-level
      // element.
      if (isset($content['#attributes'])) {
        $build['#attributes'] = $content['#attributes'];
        unset($content['#attributes']);
      }

      // Make sure block's #attached is part of the page assets.
      if (isset($content['#attached'])) {
        $build['#attached'] = $content['#attached'];
        unset($content['#attached']);
      }

      // Hide contextual links for inline blocks until the UX issues surrounding
      // editing them directly are resolved.
      // @see https://www.drupal.org/project/drupal/issues/3075308
      if (!$block instanceof InlineBlock && !empty($content['#contextual_links'])) {
        $build['#contextual_links'] = $content['#contextual_links'];
      }
      $build['content'] = $content;

      if ($event->inPreview()) {
        if ($block instanceof PreviewFallbackInterface) {
          $preview_fallback_string = $block->getPreviewFallbackString();
        }
        else {
          $preview_fallback_string = $this->t('"@block" block', ['@block' => $block->label()]);
        }
        // @todo Use new label methods so
        //   data-layout-content-preview-placeholder-label doesn't have to use
        //   preview fallback in https://www.drupal.org/node/2025649.
        $build['#attributes']['data-layout-content-preview-placeholder-label'] = $preview_fallback_string;

        if ($is_content_empty && $is_placeholder_ready) {
          $build['content']['#markup'] = $this->t(
            'Placeholder for the @preview_fallback',
            [
              '@preview_fallback' => $block->getPreviewFallbackString(),
            ]
          );
        }
      }

      $event->setBuild($build);
    }
  }

  /**
   * Get the current node id.
   *
   * @return int
   *   Returns the current node id.
   */
  protected function getCurrentNodeId(): int {
    $node = $this->routeMatch->getParameter('node');

    return ($node instanceof NodeInterface) ? $node->id() : 0;
  }

  /**
   * Get current taxonomy id.
   *
   * @return int
   *   Returns the current term id.
   */
  protected function getCurrentTaxonomyTermId(): int {
    $term = $this->routeMatch->getParameter('taxonomy_term');

    return ($term instanceof TermInterface) ? $term->id() : 0;
  }

}
