<?php

namespace Drupal\layout_builder_ipe;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\Theme\ThemeManagerInterface;

/**
 * Extension service for IPE.
 */
class LayoutBuilderIpeExtensions {

  /**
   * The layout builder ipe config object.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The theme manager.
   *
   * @var \Drupal\Core\Theme\ThemeManagerInterface
   */
  protected $themeManager;

  /**
   * The theme handler.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * Public constructor.
   */
  public function __construct(ConfigFactoryInterface $config_factory, ModuleHandlerInterface $module_handler, ThemeHandlerInterface $theme_handler, ThemeManagerInterface $theme_manager) {
    $this->configFactory = $config_factory;
    $this->moduleHandler = $module_handler;
    $this->themeHandler = $theme_handler;
    $this->themeManager = $theme_manager;
  }

  /**
   * See if the given module is enabled.
   *
   * @param string $module
   *   The module name.
   *
   * @return bool
   *   The config value.
   */
  public function moduleEnabled(string $module): bool {
    return $this->moduleHandler->moduleExists($module);
  }

  /**
   * See if the given theme is enabled.
   *
   * @param string $theme
   *   The theme name.
   *
   * @return bool
   *   The config value.
   */
  public function themeEnabled(string $theme): bool {
    return $this->themeHandler->themeExists($theme);
  }

  /**
   * See if the given theme is an admin theme.
   *
   * @param string $theme
   *   The theme name.
   *
   * @return bool
   *   TRUE if the given theme is set to be the admin theme.
   */
  public function isAdminTheme(string $theme): bool {
    return $this->configFactory->get('system.theme')->get('admin') == $theme;
  }

  /**
   * Get the name of the active theme.
   *
   * @return string
   *   The name of currently active theme.
   */
  public function getActiveTheme(): string {
    return $this->themeManager->getActiveTheme()->getName();
  }

  /**
   * Get the module handler service.
   *
   * @return \Drupal\Core\Extension\ModuleHandlerInterface
   *   The module handler service.
   */
  public function moduleHandler() {
    return $this->moduleHandler;
  }

}
