<?php

namespace Drupal\layout_builder_ipe;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\TempStore\SharedTempStoreFactory;
use Drupal\layout_builder\LayoutTempstoreRepository;
use Drupal\layout_builder\SectionStorageInterface;
use Drupal\layout_builder_ipe\Traits\SectionStorageTrait;

/**
 * Provide a tempstore repository for Layout Builder.
 *
 * This class basically wraps LayoutTempstoreRepository with the main
 * difference of overriding some key methods to be able to separate layout
 * sessions by user.
 */
class LayoutBuilderIpeTempstoreRepository extends LayoutTempstoreRepository {

  use SectionStorageTrait;

  /**
   * The shared tempstore factory.
   *
   * @var \Drupal\Core\TempStore\SharedTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $entityDisplayRepository;

  /**
   * The current user.
   *
   * @var Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The layout builder ipe config object.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $config;

  /**
   * LayoutBuilderIpeTempstoreRepository constructor.
   *
   * @param \Drupal\Core\TempStore\SharedTempStoreFactory $temp_store_factory
   *   The shared tempstore factory.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository
   *   The entity display repository.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(SharedTempStoreFactory $temp_store_factory, EntityDisplayRepositoryInterface $entity_display_repository, AccountProxyInterface $current_user, ConfigFactoryInterface $config_factory) {
    $this->tempStoreFactory = $temp_store_factory;
    $this->entityDisplayRepository = $entity_display_repository;
    $this->currentUser = $current_user;
    $this->config = $config_factory->get('layout_builder_ipe.settings');
  }

  /**
   * Returns the metadata associated with a particular section storage.
   *
   * @param \Drupal\layout_builder\SectionStorageInterface $section_storage
   *   The section storage.
   *
   * @return \Drupal\Core\TempStore\Lock|null
   *   An object with the owner and updated time if the key has a value, or
   *   NULL otherwise.
   */
  public function getMetaData(SectionStorageInterface $section_storage) {
    $key = $this->getKey($section_storage);
    return $this->getTempstore($section_storage)->getMetadata($key);
  }

  /**
   * {@inheritdoc}
   */
  protected function getKey(SectionStorageInterface $section_storage) {
    $key = parent::getKey($section_storage);
    $key .= '::' . $this->currentUser->id();
    $key = $this->checkKeyForFullViewMode($key, $section_storage);
    return $key;
  }

  /**
   * Check the tempstore key for the presence of the 'full' view mode.
   *
   * Somehow the 'full' view mode might be used even if it's not configured,
   * which creates issues for the frontend-facing discard links that would use
   * the 'default' view mode in this case.
   *
   * @param string $key
   *   The tempstore key.
   * @param \Drupal\layout_builder\SectionStorageInterface $section_storage
   *   The section storage.
   *
   * @return string
   *   The modified key or the original.
   */
  private function checkKeyForFullViewMode($key, SectionStorageInterface $section_storage) {
    if (!str_contains($key, '.full.')) {
      return $key;
    }
    $entity = $this->getEntityFromSectionStorage($section_storage);
    if (!$entity) {
      return $key;
    }
    $available_view_modes = $this->entityDisplayRepository->getViewModeOptionsByBundle($entity->getEntityTypeId(), $entity->bundle());
    if (!array_key_exists('full', $available_view_modes)) {
      $key = str_replace('.full.', '.default.', $key);
    }
    return $key;
  }

}
