<?php

namespace Drupal\layout_builder_perms_node\Plugin\Derivative;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactory;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\Utility\Error;
use Drupal\layout_builder_perms\LayoutBuilderPermissionPluginDeriverBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines per content type LayoutBuilderPermission plugins.
 */
class NodeLayoutBuilderPermissions extends LayoutBuilderPermissionPluginDeriverBase implements ContainerDeriverInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $bundleInfo;

  /**
   * The logger channel factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactory
   */
  protected $loggerFactory;

  /**
   * NodeLayoutBuilderPermissions constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $bundle_info
   *   The entity type bundle info service.
   * @param \Drupal\Core\Logger\LoggerChannelFactory $logger_factory
   *   The logger channel factory.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityTypeBundleInfoInterface $bundle_info, LoggerChannelFactory $logger_factory) {
    $this->entityTypeManager = $entity_type_manager;
    $this->bundleInfo = $bundle_info;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    // Per content type permissions.
    foreach ($this->getLayoutBuilderEnabledContentTypes() as $bundle => $bundle_info) {
      foreach ($this->getLayoutBuilderOperations() as $operation) {
        [$component, $action] = explode('_', $operation);
        $plugin_id = $bundle . ':' . $operation;
        $this->derivatives[$plugin_id] = $base_plugin_definition;
        $this->derivatives[$plugin_id]['permission'] =
          $action . ' ' . $component . 's on ' . $bundle . ' nodes';
        $this->derivatives[$plugin_id]['label'] =
          ucfirst($action) . ' ' . $component . 's on ' . $bundle_info['label'] . ' nodes';
        $this->derivatives[$plugin_id]['description'] = '';
        $this->derivatives[$plugin_id]['operation'] = $operation;
        $this->derivatives[$plugin_id]['action'] = $action;
        $this->derivatives[$plugin_id]['component'] = $component;
        $this->derivatives[$plugin_id]['entity_type'] = 'node';
        $this->derivatives[$plugin_id]['bundle'] = $bundle;
      }
    }
    return $this->derivatives;
  }

  /**
   * Returns the content types that have layout builder overrides enabled.
   *
   * @return array
   *   Array of content type definitions.
   */
  protected function getLayoutBuilderEnabledContentTypes(): array {
    $content_types = [];

    try {
      /** @var \Drupal\layout_builder\Entity\LayoutEntityDisplayInterface[] $entity_displays */
      $entity_displays = $this->entityTypeManager->getStorage('entity_view_display')
        ->loadByProperties([
          'targetEntityType' => 'node',
          'third_party_settings.layout_builder.allow_custom' => TRUE,
        ]);
      foreach ($entity_displays as $entity_display) {
        $bundle = $entity_display->getTargetBundle();
        if (!isset($content_types[$bundle])) {
          $content_types[$bundle] = $this->bundleInfo->getBundleInfo('node')[$bundle];
        }
      }
    } catch (PluginNotFoundException | InvalidPluginDefinitionException $e) {
      Error::logException($this->loggerFactory->get('layout_builder_perms'), $e);
    }

    return $content_types;
  }

}
