<?php

namespace Drupal\layout_builder_perms;

use Drupal\Component\Uuid\Uuid;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\layout_builder\SectionStorageInterface;

/**
 * Defines a pre render method for layout builder elements.
 */
class LayoutBuilderElement implements TrustedCallbackInterface {

  /**
   * Removes elements to which the current user does not have access.
   *
   * @param array $element
   *   The layout builder render element.
   *
   * @return array
   *   The modified layout builder render element.
   */
  public static function preRender(array $element): array {
    // Check each section in the layout.
    foreach (Element::children($element['layout_builder']) as $sid) {
      $section = $element['layout_builder'][$sid];

      // Check 'add section' links.
      if (isset($section['link'])) {
        if (static::elementAccess($section['link']) === FALSE) {
          unset($element['layout_builder'][$sid]);
        }
      }

      // Check 'edit section' and 'delete section' links.
      if (isset($section['#type']) && $section['#type'] == 'container') {
        foreach (Element::children($section) as $name) {
          $item = $section[$name];
          if (isset($item['link'])) {
            if (static::elementAccess($item['link']) === FALSE) {
              unset($element['layout_builder'][$sid][$name]);
            }
          }
          elseif (isset($item['#url'])) {
            if (static::elementAccess($item) === FALSE) {
              unset($element['layout_builder'][$sid][$name]);
            }
          }
        }
      }

      if (isset($section['layout-builder__section'])) {
        // Check each region in the section.
        foreach (Element::children($section['layout-builder__section']) as $rid) {
          $region = $section['layout-builder__section'][$rid];
          if (!is_array($region)) {
            continue;
          }
          $region = array_filter($region, 'is_array');

          // Check each block in the region.
          foreach (Element::children($region) as $id) {
            // Check if moving the block is allowed.
            if (Uuid::isValid($id) && !static::allowBlockMove($element['#section_storage'])) {
              // Remove block reordering from this block.
              $class = array_search('js-layout-builder-block', $region[$id]['#attributes']['class']);
              if (isset($region[$id]['#attributes']['class'][$class])) {
                unset($element['layout_builder'][$sid]['layout-builder__section'][$rid][$id]['#attributes']['class'][$class]);
              }
              // Allow for CSS overrides when the block cannot be moved.
              $element['layout_builder'][$sid]['layout-builder__section'][$rid][$id]['#attributes']['class'][] = 'layout-builder-block--no-reorder';
            }
          }

          // Check the 'add block' link for the section.
          if (isset($region['layout_builder_add_block'])) {
            if (static::elementAccess($region['layout_builder_add_block']['link']) === FALSE) {
              unset($element['layout_builder'][$sid]['layout-builder__section'][$rid]['layout_builder_add_block']);
            }
          }
        }
      }
    }

    // Attach library with CSS overrides.
    $element['layout_builder']['#attached']['library'][] = 'layout_builder_perms/css.override';

    return $element;
  }

  /**
   * Checks whether user has access to a link element.
   *
   * @param array $element
   *   The link element.
   *
   * @return bool
   *   TRUE user has access to given route, FALSE otherwise.
   */
  public static function elementAccess(array $element): bool {
    $route_name = $element['#url']->getRouteName();
    $route_params = $element['#url']->getRouteParameters();

    // Check if user has access to a named route.
    $access = \Drupal::service('access_manager')->checkNamedRoute($route_name, $route_params);

    return ($access instanceof AccessResultInterface) ? $access->isAllowed() : $access;
  }

  /**
   * Checks whether a block move is allowed.
   *
   * @param \Drupal\layout_builder\SectionStorageInterface $region_storage
   *   The section storage.
   *
   * @return bool
   *   TRUE if the current user is allowed to move the block, FALSE otherwise.
   */
  protected static function allowBlockMove(SectionStorageInterface $region_storage): bool {
    $access = \Drupal::service('access_manager')->checkNamedRoute(
      'layout_builder.move_block',
      [
        'section_storage_type' => $region_storage->getStorageType(),
        'section_storage' => $region_storage->getStorageId(),
      ],
    );

    return ($access instanceof AccessResultInterface) ? $access->isAllowed() : $access;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks(): array {
    return ['preRender'];
  }

}
