<?php

namespace Drupal\layout_builder_perms;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Plugin\ContextAwarePluginInterface;
use Drupal\Core\Plugin\ContextAwarePluginTrait;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Utility\Error;

/**
 * Defines a base class for LayoutBuilderPermission plugins.
 */
abstract class LayoutBuilderPermissionBase extends PluginBase implements ContextAwarePluginInterface, LayoutBuilderPermissionInterface {

  use ContextAwarePluginTrait;

  /**
   * {@inheritdoc}
   */
  public function getPermission(): string {
    return $this->pluginDefinition['permission'];
  }

  /**
   * {@inheritdoc}
   */
  public function getLabel(): string {
    return $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string {
    return $this->pluginDefinition['description'];
  }

  /**
   * {@inheritdoc}
   */
  public function applies(): bool {
    // By default only check whether this plugin is defined for the given
    // operation. Implementing classes can extend on this by checking other
    // context information.
    try {
      $operation = $this->getContextValue('operation');
      return $operation === $this->pluginDefinition['operation'];
    } catch (PluginException $e) {
      Error::logException(\Drupal::logger('layout_builder_perms'), $e);
      // Do not apply if the context cannot be read.
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access(string $operation, AccountInterface $account): AccessResultInterface {
    // Allow access if the given user has the permission defined by this plugin.
    return AccessResult::allowedIfHasPermission($account, $this->getPermission());
  }

}
