<?php

namespace Drupal\layout_builder_perms_block_types_per_layout\Plugin\Derivative;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Layout\LayoutPluginManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactory;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\Utility\Error;
use Drupal\layout_builder_perms\LayoutBuilderPermissionPluginDeriverBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines inline block type in layout LayoutBuilderPermission plugins.
 */
class InlineBlockTypeInLayoutPermissions extends LayoutBuilderPermissionPluginDeriverBase implements ContainerDeriverInterface {

  /**
   * The layout plugin manager.
   *
   * @var \Drupal\Core\Layout\LayoutPluginManagerInterface
   */
  protected $layoutPluginManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The logger channel factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactory
   */
  protected $loggerFactory;

  /**
   * LayoutLayoutBuilderPermissions constructor.
   *
   * @param \Drupal\Core\Layout\LayoutPluginManagerInterface $layout_plugin_manager
   *   The layout plugin manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactory $logger_factory
   *   The logger channel factory.
   */
  public function __construct(LayoutPluginManagerInterface $layout_plugin_manager, EntityTypeManagerInterface $entity_type_manager, LoggerChannelFactory $logger_factory) {
    $this->layoutPluginManager = $layout_plugin_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $container->get('plugin.manager.core.layout'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition): array {
    try {
      /** @var \Drupal\block_content\BlockContentTypeInterface[] $block_content_types */
      $block_content_types = $this->entityTypeManager->getStorage('block_content_type')->loadMultiple();
    } catch (PluginNotFoundException | InvalidPluginDefinitionException $e) {
      Error::logException($this->loggerFactory->get('layout_builder_perms'), $e);
      return $this->derivatives;
    }

    // Per layout.
    foreach ($this->layoutPluginManager->getLayoutOptions() as $layout_group) {
      foreach ($layout_group as $layout_id => $layout_name) {
        if ($layout_name) {
          // Per block type.
          foreach ($block_content_types as $block_type_id => $type) {
            // Per operation.
            foreach (['block_reorder', 'block_add', 'block_config', 'block_remove'] as $operation) {
              [$component, $action] = explode('_', $operation);
              $plugin_id = $layout_id . ':' . $block_type_id . ':' . $operation;
              $this->derivatives[$plugin_id] = $base_plugin_definition;
              $this->derivatives[$plugin_id]['permission'] =
                $action . ' ' . $block_type_id . ' ' . $component . 's in ' . $layout_id . ' layouts';
              $this->derivatives[$plugin_id]['label'] =
                ucfirst($action) . ' ' . $type->label() . ' ' . $component . 's in ' . $layout_name->getUntranslatedString() . ' layouts';
              $this->derivatives[$plugin_id]['description'] = '';
              $this->derivatives[$plugin_id]['operation'] = $operation;
              $this->derivatives[$plugin_id]['action'] = $action;
              $this->derivatives[$plugin_id]['block_type'] = $block_type_id;
              $this->derivatives[$plugin_id]['component'] = $component;
              $this->derivatives[$plugin_id]['layout'] = $layout_id;
            }
          }
        }
      }
    }
    return $this->derivatives;
  }

}
