<?php

namespace Drupal\layout_builder_perms_layout_per_bundle\Plugin\Derivative;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Layout\LayoutPluginManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactory;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\Utility\Error;
use Drupal\layout_builder_perms\LayoutBuilderPermissionPluginDeriverBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines per entity type and entity bundle LayoutBuilderPermission plugins.
 */
class EntityLayoutBuilderPermissions extends LayoutBuilderPermissionPluginDeriverBase implements ContainerDeriverInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $bundleInfo;

  /**
   * The layout plugin manager.
   *
   * @var \Drupal\Core\Layout\LayoutPluginManagerInterface
   */
  protected $layoutPluginManager;

  /**
   * The logger channel factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactory
   */
  protected $loggerFactory;

  /**
   * NodeLayoutBuilderPermissions constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $bundle_info
   *   The entity type bundle info service.
   * @param \Drupal\Core\Layout\LayoutPluginManagerInterface $layout_plugin_manager
   *   The layout plugin manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactory $logger_factory
   *   The logger channel factory.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityTypeBundleInfoInterface $bundle_info, LayoutPluginManagerInterface $layout_plugin_manager, LoggerChannelFactory $logger_factory) {
    $this->entityTypeManager = $entity_type_manager;
    $this->bundleInfo = $bundle_info;
    $this->layoutPluginManager = $layout_plugin_manager;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('plugin.manager.core.layout'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    // Per entity type and entity bundle permissions for layouts.
    $layouts = $this->getAvailableLayouts();
    foreach ($this->getLayoutBuilderEnabledEntityBundles() as $entity_type_id => $entity_bundles) {
      foreach ($entity_bundles as $bundle => $bundle_info) {
        foreach ($layouts as $layout_id => $layout_name) {
          foreach (['section_add', 'section_edit', 'section_remove'] as $operation) {
            [, $action] = explode('_', $operation);
            $plugin_id = $entity_type_id . ':' . $bundle . ':' . $layout_id . ':' . $operation . ':' . $action;
            $this->derivatives[$plugin_id] = $base_plugin_definition;
            $this->derivatives[$plugin_id]['permission'] =
              $action . ' ' . $layout_id . ' layouts on ' . $bundle . ' ' . $entity_type_id . ' entities';
            $this->derivatives[$plugin_id]['label'] =
              ucfirst($action) . ' ' . $layout_name . ' layouts on ' . $bundle_info['label'] . ' ' . strtolower($bundle_info['entity_type_label']) . ' entities';
            $this->derivatives[$plugin_id]['description'] = '';
            $this->derivatives[$plugin_id]['operation'] = $operation;
            $this->derivatives[$plugin_id]['action'] = $action;
            $this->derivatives[$plugin_id]['layout'] = $layout_id;
            $this->derivatives[$plugin_id]['entity_type'] = $entity_type_id;
            $this->derivatives[$plugin_id]['bundle'] = $bundle;
          }
        }
      }
    }
    return $this->derivatives;
  }

  /**
   * Returns the entity bundles that have layout builder overrides enabled.
   *
   * @return array
   *   Array of entity bundle definitions, keyed first by entity type ID and
   *   then entity bundle name.
   */
  protected function getLayoutBuilderEnabledEntityBundles(): array {
    $entity_bundles = [];

    try {
      /** @var \Drupal\layout_builder\Entity\LayoutEntityDisplayInterface[] $entity_displays */
      $entity_displays = $this->entityTypeManager->getStorage('entity_view_display')
        ->loadByProperties(['third_party_settings.layout_builder.allow_custom' => TRUE]);
      foreach ($entity_displays as $entity_display) {
        $type = $entity_display->getTargetEntityTypeId();
        $bundle = $entity_display->getTargetBundle();
        if (!isset($entity_bundles[$type][$bundle])) {
          $entity_bundles[$type][$bundle] = $this->bundleInfo->getBundleInfo($type)[$bundle];
          // Add the entity type label.
          $entity_type_label = $this->entityTypeManager->getDefinition($type)->getLabel()->getUntranslatedString();
          $entity_bundles[$type][$bundle]['entity_type_label'] = $entity_type_label;
        }
      }
    } catch (PluginNotFoundException | InvalidPluginDefinitionException $e) {
      Error::logException($this->loggerFactory->get('layout_builder_perms'), $e);
    }

    return $entity_bundles;
  }

  /**
   * Returns the available layouts.
   *
   * @return array
   *   An array of layout names keyed by layout IDs.
   */
  protected function getAvailableLayouts(): array {
    $layouts = [];

    foreach ($this->layoutPluginManager->getLayoutOptions() as $layout_group) {
      foreach ($layout_group as $layout_id => $layout_name) {
        if ($layout_name) {
          /** @var \Drupal\Core\StringTranslation\TranslatableMarkup $layout_name */
          $layouts[$layout_id] = $layout_name->getUntranslatedString();
        }
      }
    }

    return $layouts;
  }

}
