<?php

namespace Drupal\layout_builder_perms\Access;

use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Component\Plugin\Exception\MissingValueContextException;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Logger\LoggerChannelFactory;
use Drupal\Core\Plugin\Context\ContextHandlerInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Drupal\Core\Plugin\ContextAwarePluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Utility\Error;
use Drupal\layout_builder\LayoutTempstoreRepositoryInterface;
use Drupal\layout_builder\Section;
use Drupal\layout_builder\SectionStorageInterface;
use Drupal\layout_builder_perms\AccessManagerInterface;
use Drupal\layout_builder_perms\Event\LayoutBuilderPermissionPluginContexts;
use Drupal\layout_builder_perms\LayoutBuilderPermissionInterface;
use Drupal\layout_builder_perms\LayoutBuilderPermissionPluginManagerInterface;

/**
 * Defines a class to check access to layout builder functionality.
 */
class AccessManager implements AccessManagerInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The layout builder permissions plugin manager.
   *
   * @var \Drupal\layout_builder_perms\LayoutBuilderPermissionPluginManagerInterface
   */
  protected $layoutBuilderPermissions;

  /**
   * The context repository.
   *
   * @var \Drupal\Core\Plugin\Context\ContextRepositoryInterface
   */
  protected $contextRepository;

  /**
   * The context handler.
   *
   * @var \Drupal\Core\Plugin\Context\ContextHandlerInterface
   */
  protected $contextHandler;

  /**
   * The layout tempstore repository.
   *
   * @var \Drupal\layout_builder\LayoutTempstoreRepositoryInterface
   */
  protected $layoutTempstoreRepository;

  /**
   * The logger channel factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactory
   */
  protected $loggerFactory;

  /**
   * AccessManager constructor.
   *
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\layout_builder_perms\LayoutBuilderPermissionPluginManagerInterface $layout_builder_permissions
   *   The layout builder permissions plugin manager.
   * @param \Drupal\Core\Plugin\Context\ContextRepositoryInterface $context_repository
   *   The context repository.
   * @param \Drupal\Core\Plugin\Context\ContextHandlerInterface $context_handler
   *   The context handler.
   * @param \Drupal\layout_builder\LayoutTempstoreRepositoryInterface $layout_tempstore_repository
   *   The layout tempstore repository.
   * @param \Drupal\Core\Logger\LoggerChannelFactory $logger_factory
   *   The logger channel factory.
   */
  public function __construct(AccountInterface $current_user, LayoutBuilderPermissionPluginManagerInterface $layout_builder_permissions, ContextRepositoryInterface $context_repository, ContextHandlerInterface $context_handler, LayoutTempstoreRepositoryInterface $layout_tempstore_repository, LoggerChannelFactory $logger_factory) {
    $this->currentUser = $current_user;
    $this->layoutBuilderPermissions = $layout_builder_permissions;
    $this->contextRepository = $context_repository;
    $this->contextHandler = $context_handler;
    $this->layoutTempstoreRepository = $layout_tempstore_repository;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public function access(string $operation, RouteMatchInterface $route_match = NULL, SectionStorageInterface $section_storage = NULL, AccountInterface $account = NULL, $return_as_object = FALSE) {
    if (NULL === $account) {
      $account = $this->currentUser;
    }

    // Make sure we use the temp store section storage to check access for new
    // sections not yet saved in the entityq.
    if ($section_storage) {
      $section_storage = $this->layoutTempstoreRepository->get($section_storage);
    }

    // Check plugin defined permissions.
    $result = AccessResult::allowed();

    // Create a list of filters to only load plugins we need.
    [$component, $action] = explode('_', $operation);
    $layout = $route_match ? $this->getLayoutFromRouteMatch($route_match, $section_storage) : NULL;
    try {
      $entity = $section_storage->getContextValue('entity');
    }
    catch (ContextException $e) {
      $entity = NULL;
    }

    if ($entity) {
      // Override the action if the layout has a third party setting.
      $action = $layout ? ($layout->getThirdPartySetting('layout_builder_perms', 'action', '') ?: $action) : $action;
      // Rebuild the operation string in case the action has changed. This is
      // especially relevant for the 'add' action, because this is never
      // actually called with $operation = 'section_add'.
      $operation = $component . '_' . $action;

      $filters = [
        'operation' => $operation,
        'component' => $component,
        'action' => $action,
        'layout' => $layout ? $layout->getLayoutId() : NULL,
        'block_type' => $route_match ? $this->getBlockTypeFromRouteMatch($route_match) : NULL,
        'entity_type' => $entity ? $entity->getEntityTypeId() : NULL,
        'bundle' => $entity ? $entity->bundle() : NULL,
      ];
      foreach ($this->layoutBuilderPermissions->getPermissionPlugins(array_filter($filters)) as $permission_plugin) {
        // Set the plugin context if the plugin is context aware.
        if ($permission_plugin instanceof ContextAwarePluginInterface) {
          $this->setPluginContext($permission_plugin, $operation, $route_match, $section_storage);
        }

        // Check access if the plugin applies.
        if ($permission_plugin->applies()) {
          $result = $result->andIf($permission_plugin->access($operation, $account));
        }
      }
    }

    return $return_as_object ? $result : $result->isAllowed();
  }

  /**
   * Sets the context for a permission plugin.
   *
   * @param \Drupal\Core\Plugin\ContextAwarePluginInterface $plugin
   *   The permission plugin to set the context for.
   * @param string $operation
   *   The operation to check access for.
   * @param \Drupal\Core\Routing\RouteMatchInterface|null $route_match
   *   (optional) The route match of the route to check access for.
   * @param \Drupal\layout_builder\SectionStorageInterface|null $section_storage
   *   (optional) The section storage for the layout that is being edited.
   */
  protected function setPluginContext(ContextAwarePluginInterface $plugin, string $operation, RouteMatchInterface $route_match = NULL, SectionStorageInterface $section_storage = NULL): void {
    // Pass in the operation.
    try {
      // The getContextDefinition() method will throw a ContextException when
      // the 'operation' context is not set. We use this to skip setting the
      // context value in that case.
      $plugin->getContextDefinition('operation');
      $plugin->setContextValue(
        'operation',
        $operation
      );
    } catch (ContextException $e) {
      // Skip if the plugin has no 'operation' context.
    }

    // Pass in the layout.
    try {
      // The getContextDefinition() method will throw a ContextException when
      // the 'layout' context is not set. We use this to skip setting the
      // context value in that case.
      $plugin->getContextDefinition('layout');

      // Find the layout that is being edited.
      if ($route_match !== NULL) {
        $layout = $this->getLayoutFromRouteMatch($route_match, $section_storage);
        if (isset($layout)) {
          // Set the context value.
          $plugin->setContextValue(
            'layout',
            $layout
          );
        }
      }
    } catch (ContextException $e) {
      // Skip if the plugin has no 'layout' context.
    }

    // Pass in the entity.
    try {
      // The getContextDefinition() method will throw a ContextException when
      // the 'entity' context is not set. We use this to skip setting the
      // context value in that case.
      $plugin->getContextDefinition('entity');

      $entity = $section_storage->getContextValue('entity');
      if ($entity instanceof EntityInterface) {
        $plugin->setContextValue(
          'entity',
          $entity
        );
      }
    } catch (ContextException $e) {
      // Skip if the plugin has no 'entity' context.
    }

    // Inject runtime contexts. This is necessary to get the node context.
    $contexts = $this->contextRepository->getRuntimeContexts($plugin->getContextMapping());
    try {
      $this->contextHandler->applyContextMapping($plugin, $contexts);
    } catch (MissingValueContextException | ContextException $e) {
      Error::logException($this->loggerFactory->get('layout_builder_perms'), $e);
    }

    // Allow other modules to add additional contexts.
    if ($plugin instanceof LayoutBuilderPermissionInterface) {
      $event = new LayoutBuilderPermissionPluginContexts($plugin, $operation, $route_match, $section_storage);
      \Drupal::service('event_dispatcher')->dispatch($event);
    }
  }

  /**
   * Get a layout from the route match and section storage.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface|null $route_match
   *   The route match of the route to get the layout for.
   * @param \Drupal\layout_builder\SectionStorageInterface|null $section_storage
   *   (optional) The section storage for the layout that is being edited.
   */
  protected function getLayoutFromRouteMatch(RouteMatchInterface $route_match, SectionStorageInterface $section_storage = NULL) {
    switch ($route_match->getRouteName()) {
      // When inserting a new section the layout type is determined by the
      // plugin ID.
      case 'layout_builder.add_section':
        $plugin_id = $route_match->getParameter('plugin_id');
        if ($plugin_id !== NULL) {
          return Section::fromArray(['layout_id' => $plugin_id]);
        }
        break;

      case 'layout_builder.configure_section':
        // When selecting the layout for a section the layout type is
        // determined by the plugin ID.
        $plugin_id = $route_match->getParameter('plugin_id');
        if ($plugin_id !== NULL) {
          // Create a layout section that can be injected as context. We add
          // a third party setting to allow permission plugins to
          // differentiate between selecting a layout for the section and
          // configuring an existing section.
          return Section::fromArray([
            'layout_id' => $plugin_id,
            'third_party_settings' => ['layout_builder_perms' => ['action' => 'add']],
          ]);
        } else {
          // When configuring an existing section the layout is determined
          // by the delta of the section in the section storage.
          $delta = $route_match->getParameter('delta');
          if ($delta !== NULL) {
            try {
              // Get the layout from the section storage.
              return $section_storage->getSection($delta);
            } catch (\OutOfBoundsException $e) {
              // We are probably checking access to the route for adding a
              // new section. The layout is therefore undetermined at this
              // point.
            }
          }
        }
        break;

      case 'layout_builder.choose_block':
      case 'layout_builder.add_block':
      case 'layout_builder.choose_inline_block':
      case 'layout_builder.update_block':
      case 'layout_builder.move_block_form':
      case 'layout_builder.move_block':
      case 'layout_builder.remove_block':
      case 'layout_builder.remove_section':
        // The layout is determined by the delta of the section in the
        // section storage.
        $delta = $route_match->getParameter('delta') ?? $route_match->getParameter('delta_to');
        if ($delta !== NULL) {
          try {
            // Get the layout from the section storage.
            return $section_storage->getSection($delta);
          } catch (\OutOfBoundsException $e) {
            // The layout is undetermined at this point.
          }
        }
        break;
    }

    return NULL;
  }

  /**
   * Get a block type from the route match.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface|null $route_match
   *   The route match of the route to get the block type for.
   */
  protected function getBlockTypeFromRouteMatch(RouteMatchInterface $route_match) {
    if ($route_match && 'layout_builder.add_block' === $route_match->getRouteName() && $plugin_id = $route_match->getParameter('plugin_id')) {
      if (strpos($plugin_id, ':') !== FALSE) {
        [, $block_type] = explode(':', $plugin_id);
        return $block_type;
      }
    }
    return NULL;
  }

}
