<?php

namespace Drupal\layout_builder_perms\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Cache\RefinableCacheableDependencyInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\layout_builder\SectionStorageInterface;
use Drupal\layout_builder_perms\AccessManagerInterface;

/**
 * Override Layout Builder access check.
 *
 * @ingroup layout_builder_access
 *
 * @internal
 *   Tagged services are internal.
 */
class AdvancedAccessCheck implements AccessInterface {

  /**
   * The access manager.
   *
   * @var \Drupal\layout_builder_perms\Access\AccessManager
   */
  protected $accessManager;

  /**
   * AdvancedAccessCheck constructor.
   *
   * @param \Drupal\layout_builder_perms\Access\AccessManager $access_manager
   *   The access manager.
   */
  public function __construct(AccessManager $access_manager) {
    $this->accessManager = $access_manager;
  }

  /**
   * Checks routing access to the layout.
   *
   * @param \Drupal\layout_builder\SectionStorageInterface $section_storage
   *   The section storage.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match for the route.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(SectionStorageInterface $section_storage, AccountInterface $account, RouteMatchInterface $route_match) {
    $operation = $route_match->getRouteObject()->getRequirement('_layout_builder_perms_access');
    $access = $section_storage->access($operation, $account, TRUE);

    // Check access to layout builder operations covered by plugin permissions.
    if (in_array($operation, AccessManagerInterface::LAYOUT_BUILDER_OPERATIONS)) {
      $access = $this->accessManager->access($operation, $route_match, $section_storage, $account, TRUE);
    }

    // Check for the global permission unless the section storage checks
    // permissions itself.
    if (!$section_storage->getPluginDefinition()->get('handles_permission_check')) {
      $access = $access->andIf(AccessResult::allowedIfHasPermission($account, 'configure any layout'));
    }

    if ($access instanceof RefinableCacheableDependencyInterface) {
      $access->addCacheableDependency($section_storage);
    }

    return $access;
  }

}
