<?php

namespace Drupal\layout_builder_perms\EventSubscriber;

use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContextAwarePluginInterface;
use Drupal\layout_builder_perms\Event\LayoutBuilderPermissionPluginContexts;
use Drupal\layout_builder_perms\LayoutBuilderPermissionPluginContextProviderBase;

/**
 * Adds a content block type context to layout builder permission plugins.
 */
class ContentBlockTypePluginContext extends LayoutBuilderPermissionPluginContextProviderBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * ContentBlockTypePluginContext constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  function addPluginContexts(LayoutBuilderPermissionPluginContexts $event): void {
    $plugin = $event->getPlugin();

    if ($plugin instanceof ContextAwarePluginInterface) {
      // Pass in the block.
      try {
        // The getContextDefinition() method will throw a ContextException when
        // the 'block' context is not set. We use this to skip setting the
        // context value in that case.
        $block_type_context_definition = $plugin->getContextDefinition('block');

        // Find the block that is being manipulated.
        $route_match = $event->getRouteMatch();
        $section_storage = $event->getSectionStorage();
        if ($route_match !== NULL) {
          // Parameters for update / move / remove block routes.
          $delta = $route_match->getParameter('delta') ?? $route_match->getParameter('delta_from');
          $uuid = $route_match->getParameter('uuid') ?? $route_match->getParameter('block_uuid');
          // Parameter for choose inline block route.
          $plugin_id = $route_match->getParameter('plugin_id');
          if ($delta !== NULL && $uuid !== NULL) {
            $layout = $section_storage->getSection($delta);
            /** @var \Drupal\layout_builder\Plugin\Block\InlineBlock $block_plugin */
            $block_plugin = $layout->getComponent($uuid)->getPlugin();
            if (!empty($block_plugin->getConfiguration()['block_serialized'])) {
              $block = unserialize($block_plugin->getConfiguration()['block_serialized']);
            }
            elseif (!empty($block_plugin->getConfiguration()['block_revision_id'])) {
              $block = $this->entityTypeManager->getStorage('block_content')->loadRevision($block_plugin->getConfiguration()['block_revision_id']);
            }
          }
          elseif ($plugin_id !== NULL) {
            if (strpos($plugin_id, ':') !== FALSE) {
              [, $block_type] = explode(':', $plugin_id);
              if ($block_type) {
                $block = $this->entityTypeManager->getStorage('block_content')
                  ->create(['type' => $block_type]);
              }
            }
          }

          // Set the context value.
          if (!empty($block)) {
            $plugin->setContextValue(
              'block',
              $block
            );
          }
        }
      } catch (ContextException $e) {
        // Skip if the plugin has no 'block' context.
      }
    }

    $event->setPlugin($plugin);
  }

}
