<?php

namespace Drupal\layout_builder_perms;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\Utility\Error;

/**
 * Defines a plugin manager for LayoutBuilderPermission plugins.
 */
class LayoutBuilderPermissionPluginManager extends DefaultPluginManager implements LayoutBuilderPermissionPluginManagerInterface {

  /**
   * Statically cache the available plugins for a set of filters.
   *
   * @var array
   */
  protected $plugins = [];

  /**
   * Constructs a LayoutBuilderPermissionPluginManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/LayoutBuilderPermission',
      $namespaces,
      $module_handler,
      'Drupal\layout_builder_perms\LayoutBuilderPermissionInterface',
      'Drupal\layout_builder_perms\Annotation\LayoutBuilderPermission'
    );
    $this->alterInfo('layout_builder_permission_info');
    $this->setCacheBackend($cache_backend, 'layout_builder_permission_info_plugins');
  }

  /**
   * {@inheritdoc}
   */
  public function getPermissionPlugins($filters = []): array {
    $filter_cache_id = $filters ? implode(':', $filters) : '_no_filters';
    $cache_backend_id = implode(':', [$this->cacheKey, $filter_cache_id]);

    // We statically cache the instantiated plugins. Directly return them if
    // available.
    if (isset($this->plugins[$filter_cache_id])) {
      return $this->plugins[$filter_cache_id];
    }
    // Try to get the plugin definitions from the cache backend.
    elseif ($cache = $this->cacheGet($cache_backend_id)){
      $definitions = $cache->data;
    }
    else {
      // No cache found. If filters are passed, only instantiate the plugins we
      // actually need.
      if ($filters) {
        $definitions = [];
        foreach ($this->getDefinitions() as $plugin_id => $plugin_definition) {
          $applies = TRUE;
          foreach ($filters as $key => $value) {
            // When a key is not set in the plugin definition, we assume the
            // plugin applies.
            if (isset($plugin_definition[$key]) && $plugin_definition[$key] !== $value) {
              $applies = FALSE;
              break;
            }
          }
          if ($applies) {
            $definitions[$plugin_id] = $plugin_definition;
          }
        }
        // Permanently cache the (filtered) definitions for performance.
        $this->cacheSet($cache_backend_id, $definitions, Cache::PERMANENT, $this->cacheTags);
      }
      else {
        $definitions = $this->getDefinitions();
      }
    }

    // Instantiate our plugins and statically cache them.
    $this->plugins[$filter_cache_id] = [];
    foreach ($definitions as $plugin_id => $plugin_definition) {
      try {
        $this->plugins[$filter_cache_id][] = $this->createInstance($plugin_id);
      } catch (PluginException $e) {
        Error::logException(\Drupal::logger('layout_builder_perms'), $e);
      }
    }

    return $this->plugins[$filter_cache_id];
  }

}
