<?php

namespace Drupal\layout_builder_perms;

use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\layout_builder\LayoutBuilderOverridesPermissions;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides dynamic permissions for Layout Builder overrides.
 *
 * @see \Drupal\layout_builder\Plugin\SectionStorage\OverridesSectionStorage::access()
 *
 * @internal
 *   Dynamic permission callbacks are internal.
 */
class LayoutBuilderPermissions extends LayoutBuilderOverridesPermissions {

  /**
   * The LayoutBuilderPermission plugin manager.
   *
   * @var \Drupal\layout_builder_perms\LayoutBuilderPermissionPluginManager
   */
  protected $layoutBuilderPermissions;

  /**
   * LayoutBuilderOverridesPermissions constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $bundle_info
   *   The bundle info service.
   * @param \Drupal\layout_builder_perms\LayoutBuilderPermissionPluginManager $layout_builder_permissions
   *   The LayoutBuilderPermission plugin manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityTypeBundleInfoInterface $bundle_info, LayoutBuilderPermissionPluginManager $layout_builder_permissions) {
    parent::__construct($entity_type_manager, $bundle_info);

    $this->layoutBuilderPermissions = $layout_builder_permissions;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('plugin.manager.layout_builder_perms')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function permissions(): array {
    $permissions = [];

    // Global permissions on user's own content.
    /** @var \Drupal\layout_builder\Entity\LayoutEntityDisplayInterface[] $entity_displays */
    $entity_displays = $this->entityTypeManager->getStorage('entity_view_display')->loadByProperties(['third_party_settings.layout_builder.allow_custom' => TRUE]);
    foreach ($entity_displays as $entity_display) {
      $entity_type_id = $entity_display->getTargetEntityTypeId();
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      $bundle = $entity_display->getTargetBundle();
      $args = [
        '%entity_type' => $entity_type->getCollectionLabel(),
        '@entity_type_singular' => $entity_type->getSingularLabel(),
        '@entity_type_plural' => $entity_type->getPluralLabel(),
        '%bundle' => $this->bundleInfo->getBundleInfo($entity_type_id)[$bundle]['label'],
      ];
      if ($entity_type->hasKey('bundle')) {
        $permissions["configure own editable $bundle $entity_type_id layout overrides"] = [
          'title' => $this->t('%entity_type - %bundle: Configure layout overrides for @entity_type_plural that user owns', $args),
        ];
      }
    }

    // Plugin defined permissions.
    foreach ($this->layoutBuilderPermissions->getPermissionPlugins() as $permission_plugin) {
      $permission = ['title' => $permission_plugin->getLabel()];
      if (!empty($permission_plugin->getDescription())) {
        $permission['description'] = $permission_plugin->getDescription();
      }

      $permissions[$permission_plugin->getPermission()] = $permission;
    }

    return $permissions;
  }

}
