<?php

namespace Drupal\layout_builder_reusable_blocks\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for Layout Builder Reusable Blocks module.
 *
 * This form allows administrators to configure:
 * - Warning message settings for reusable blocks
 * - Block behavior settings (editing permissions and default behavior)
 */
class LayoutBuilderReusableBlocksConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'layout_builder_reusable_blocks.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'layout_builder_reusable_blocks_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Load existing configuration.
    $config = $this->config('layout_builder_reusable_blocks.settings');

    // Warning Message Settings section.
    $form['warning_message'] = [
      '#type' => 'details',
      '#title' => $this->t('Warning Message Settings'),
      '#description' => $this->t('Configure warning messages shown to users when editing reusable blocks.'),
      '#open' => TRUE,
    ];

    // Checkbox to enable/disable warning messages.
    $form['warning_message']['show_warning'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show warning message when editing reusable blocks'),
      '#description' => $this->t('When enabled, a warning message will be displayed when editing reusable blocks in Layout Builder to inform users that changes will affect all instances of the block.'),
      '#default_value' => $config->get('show_warning') ?? TRUE,
    ];

    // Textarea for customizing warning message text.
    $form['warning_message']['warning_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Warning message text'),
      '#description' => $this->t('The text to display in the warning message. This helps users understand that changes to reusable blocks affect all instances.'),
      '#default_value' => $config->get('warning_text') ?? 'This is a reusable block. Changes made here will affect all instances of this block.',
      '#rows' => 3,
      '#states' => [
        'visible' => [
          ':input[name="show_warning"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Block Behavior Settings section.
    $form['block_behavior'] = [
      '#type' => 'details',
      '#title' => $this->t('Block Behavior Settings'),
      '#description' => $this->t('Configure how reusable blocks behave in Layout Builder.'),
      '#open' => TRUE,
    ];

    // Checkbox to allow/disallow editing of reusable blocks.
    $form['block_behavior']['allow_editing_reusable_blocks'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow editing reusable blocks'),
      '#description' => $this->t('When enabled, users will be able to edit reusable blocks directly in Layout Builder. When disabled, reusable blocks cannot be edited from the Layout Builder interface.'),
      '#default_value' => $config->get('allow_editing_reusable_blocks') ?? FALSE,
    ];

    // Checkbox to make all blocks reusable by default.
    $form['block_behavior']['make_all_blocks_reusable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Make all blocks reusable by default'),
      '#description' => $this->t('When enabled, all blocks added in Layout Builder will be created as reusable blocks instead of inline blocks. This removes the inline/reusable choice from the add block interface and streamlines the workflow.'),
      '#default_value' => $config->get('make_all_blocks_reusable') ?? FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    // Save all configuration values.
    $this->config('layout_builder_reusable_blocks.settings')
      ->set('show_warning', $form_state->getValue('show_warning'))
      ->set('warning_text', $form_state->getValue('warning_text'))
      ->set('allow_editing_reusable_blocks', $form_state->getValue('allow_editing_reusable_blocks'))
      ->set('make_all_blocks_reusable', $form_state->getValue('make_all_blocks_reusable'))
      ->save();
  }

}
