<?php

namespace Drupal\layout_builder_reusable_blocks\Plugin\Block;

use Drupal\block_content\Plugin\Block\BlockContentBlock;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Custom block content block plugin for reusable blocks.
 *
 * This plugin extends the standard BlockContentBlock to add functionality
 * for managing reusable blocks in Layout Builder, including:
 * - Configuration-based access control for editing
 * - Warning messages when editing reusable blocks
 * - Custom form handling for Layout Builder context.
 */
class LayoutBuilderReusableContentBlock extends BlockContentBlock implements ContainerFactoryPluginInterface {

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    // Detect if we're in Layout Builder context by checking for specific form state values.
    $is_layout_builder = $form_state->get('layout_builder_tempstore_key') ||
                        $form_state->get('section_storage') ||
                        $form_state->getBuildInfo()['base_form_id'] === 'layout_builder_configure_block';

    // Only apply custom behavior when in Layout Builder and we have a block entity.
    if ($is_layout_builder && $this->getEntity()) {
      // Load module configuration settings.
      $config = $this->configFactory->get('layout_builder_reusable_blocks.settings');
      $allow_editing = $config->get('allow_editing_reusable_blocks') ?? FALSE;

      // If editing is disabled, prevent form access and show warning.
      if ($allow_editing) {
        // Add warning message if enabled in configuration.
        $show_warning = $config->get('show_warning') ?? TRUE;
        if ($show_warning) {
          $warning_text = $config->get('warning_text') ?? 'This is a reusable block. Changes made here will affect all instances of this block.';

          // Create warning message container with appropriate styling.
          $form['reusable_warning'] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['messages', 'messages--warning']],
            '#weight' => -100,
          ];

          $form['reusable_warning']['message'] = [
            '#markup' => $this->t($warning_text),
          ];
        }

        // Mark that we're in Layout Builder context for later form processing.
        $form_state->set('is_layout_builder', TRUE);
        $block = $this->getEntity();

        // Add the block entity form for editing block content.
        $form['block_form'] = [
          '#type' => 'container',
          '#weight' => -50,
          '#parents' => ['settings', 'block_form'],
          '#form_mode' => 'default',
        ];

        // Build the entity form using the default form mode.
        $form_mode = 'default';
        $form_display = EntityFormDisplay::collectRenderDisplay($block, $form_mode);
        $form_display->buildForm($block, $form['block_form'], $form_state);

        $form['block_form']['#form_mode'] = $form_mode;

        // Adjust label weight for better layout.
        $form['admin_label']['#weight'] = -99;
        $form['label']['#weight'] = -98;
        $form['label_display']['#weight'] = -97;

        // Hide administrative fields that shouldn't be shown in Layout Builder.
        if (isset($form['block_form']['info'])) {
          $form['block_form']['info']['#access'] = FALSE;
        }
        if (isset($form['block_form']['revision_log'])) {
          $form['block_form']['revision_log']['#access'] = FALSE;
        }
      }
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);

    // Only process custom validation if we're in Layout Builder context.
    if (!$form_state->get('is_layout_builder')) {
      return;
    }

    // Get the block entity and form display for validation.
    $block = $this->getEntity();
    $block_form = $form['block_form'];
    $form_display = EntityFormDisplay::collectRenderDisplay($block, $block_form['#form_mode']);

    // Handle subform state for proper validation context.
    $complete_form_state = $form_state instanceof SubformStateInterface ? $form_state->getCompleteFormState() : $form_state;

    // Extract and validate form values using the entity form display.
    $form_display->extractFormValues($block, $block_form, $complete_form_state);
    $form_display->validateFormValues($block, $block_form, $complete_form_state);

    // Store form parents for later use in submit handler.
    $form_state->setTemporaryValue('block_form_parents', $block_form['#parents']);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);

    // Only process custom submission if we're in Layout Builder context.
    if (!$form_state->get('is_layout_builder')) {
      return;
    }

    // Get the block form from the nested array structure.
    $block_form = NestedArray::getValue($form, $form_state->getTemporaryValue('block_form_parents'));
    $form_display = EntityFormDisplay::collectRenderDisplay($this->getEntity(), $block_form['#form_mode']);

    // Handle subform state for proper submission context.
    $complete_form_state = $form_state instanceof SubformStateInterface ? $form_state->getCompleteFormState() : $form_state;

    // Extract form values and save the block entity.
    $form_display->extractFormValues($this->getEntity(), $block_form, $complete_form_state);
    $this->getEntity()->save();
  }

}
