<?php

namespace Drupal\layout_builder_section_block_duplicate\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\layout_builder\Section;
use Drupal\layout_builder\SectionStorageInterface;
use Drupal\layout_builder\LayoutTempstoreRepositoryInterface;
use Drupal\layout_builder\Controller\LayoutRebuildTrait;

/**
 * Controller responsible for cloning sections in Layout Builder.
 */
class CloneSectionController extends ControllerBase {

  // Reuses core's helper methods for rebuilding the Layout Builder UI.
  use LayoutRebuildTrait;

  /**
   * Service that manages temporary storage of Layout Builder configurations.
   *
   * @var \Drupal\layout_builder\LayoutTempstoreRepositoryInterface
   */
  protected $layoutTempstoreRepository;

  /**
   * Constructor for dependency injection.
   *
   * @param \Drupal\layout_builder\LayoutTempstoreRepositoryInterface $layout_tempstore_repository
   *   The Layout Builder tempstore repository service.
   */
  public function __construct(LayoutTempstoreRepositoryInterface $layout_tempstore_repository) {
    $this->layoutTempstoreRepository = $layout_tempstore_repository;
  }

  /**
   * {@inheritdoc}
   *
   * Factory method used by Drupal’s service container to instantiate
   * the controller.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('layout_builder.tempstore_repository')
    );
  }

  /**
   * Clones a section and inserts it right after the original one.
   *
   * @param \Drupal\layout_builder\SectionStorageInterface $section_storage
   *   The section storage object (injected automatically by the route's
   *   parameter converter).
   * @param int $delta
   *   The index of the section to clone.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The rebuilt Layout Builder UI (Ajax/off-canvas response).
   */
  public function clone(SectionStorageInterface $section_storage, $delta) {
    $original_section = $section_storage->getSection($delta);
    if (!$original_section) {
      $this->messenger()->addError($this->t('Section not found.'));
      return $this->redirect('<current>');
    }

    $cloned_section = new Section(
      $original_section->getLayoutId(),
      $original_section->getLayoutSettings()
    );

    foreach ($original_section->getComponents() as $component) {
      $cloned_section->appendComponent(clone $component);
    }

    $section_storage->insertSection($delta + 1, $cloned_section);

    $this->layoutTempstoreRepository->set($section_storage);

    $this->messenger()->addStatus($this->t('Section cloned successfully.'));

    return $this->rebuildLayout($section_storage);
  }

}
