<?php

namespace Drupal\layout_builder_section_block_duplicate\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\layout_builder\Section;
use Drupal\layout_builder\SectionStorageInterface;
use Drupal\layout_builder\LayoutTempstoreRepositoryInterface;
use Drupal\layout_builder\Controller\LayoutRebuildTrait;
use Drupal\Component\Uuid\UuidInterface;

/**
 * Controller responsible for cloning sections in Layout Builder.
 */
class CloneSectionController extends ControllerBase {

  use LayoutRebuildTrait;

  /**
   * Service that manages temporary storage of Layout Builder configurations.
   *
   * @var \Drupal\layout_builder\LayoutTempstoreRepositoryInterface
   */
  protected $layoutTempstoreRepository;

  /**
   * Service that generates UUIDs.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected $uuidService;

  /**
   * Constructor for dependency injection.
   *
   * @param \Drupal\layout_builder\LayoutTempstoreRepositoryInterface $layout_tempstore_repository
   *   The Layout Builder tempstore repository service.
   * @param \Drupal\Component\Uuid\UuidInterface $uuid_service
   *   The UUID service.
   */
  public function __construct(
    LayoutTempstoreRepositoryInterface $layout_tempstore_repository,
    UuidInterface $uuid_service,
  ) {
    $this->layoutTempstoreRepository = $layout_tempstore_repository;
    $this->uuidService = $uuid_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('layout_builder.tempstore_repository'),
      $container->get('uuid')
    );
  }

  /**
   * Clones a section and inserts it right after the original one.
   *
   * @param \Drupal\layout_builder\SectionStorageInterface $section_storage
   *   The section storage object.
   * @param int $delta
   *   The index of the section to clone.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The rebuilt Layout Builder UI (Ajax/off-canvas response).
   */
  public function clone(SectionStorageInterface $section_storage, $delta) {
    $original_section = $section_storage->getSection($delta);
    if (!$original_section) {
      $this->messenger()->addError($this->t('Section not found.'));
      return $this->redirect('<current>');
    }

    $cloned_section = new Section(
      $original_section->getLayoutId(),
      $original_section->getLayoutSettings()
    );

    foreach ($original_section->getComponents() as $component) {
      $new_component = clone $component;

      $configuration = $new_component->get('configuration');
      if (
        !empty($configuration['provider']) &&
        $configuration['provider'] === 'layout_builder'
      ) {
        if (isset($configuration['uuid'])) {
          $configuration['uuid'] = $this->uuidService->generate();
        }

        if (isset($configuration['block_serialized'])) {
          $inline_block = unserialize($configuration['block_serialized']);
          $inline_block = $inline_block->createDuplicate();

          $uuid = $this->uuidService->generate();
          $inline_block->set('uuid', $uuid);
          $inline_block->enforceIsNew();
          $inline_block->save();

          $configuration['block_serialized'] = serialize($inline_block);
          $configuration['block_revision_id'] = $inline_block->getRevisionId();
        }

        $new_component->setConfiguration($configuration);
      }

      $cloned_section->appendComponent($new_component);
    }

    $section_storage->insertSection($delta + 1, $cloned_section);
    $this->layoutTempstoreRepository->set($section_storage);
    $this->messenger()->addStatus($this->t('Section cloned successfully.'));

    return $this->rebuildLayout($section_storage);
  }

}
