<?php

namespace Drupal\layout_builder_section_block_duplicate\Form;

use Drupal\Core\Ajax\AjaxFormHelperTrait;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\WorkspaceDynamicSafeFormInterface;
use Drupal\layout_builder\Context\LayoutBuilderContextTrait;
use Drupal\layout_builder\Controller\LayoutRebuildTrait;
use Drupal\layout_builder\LayoutBuilderHighlightTrait;
use Drupal\layout_builder\LayoutTempstoreRepositoryInterface;
use Drupal\layout_builder\SectionStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\layout_builder\Form\WorkspaceSafeFormTrait;
use Drupal\layout_builder\SectionComponent;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Form used to clone an individual block inside a Layout Builder section.
 */
class CloneBlockForm extends FormBase implements WorkspaceDynamicSafeFormInterface {

  use AjaxFormHelperTrait;
  use LayoutBuilderContextTrait;
  use LayoutRebuildTrait;
  use LayoutBuilderHighlightTrait;
  use WorkspaceSafeFormTrait;


  /**
   * Storage for the current section collection (entire layout).
   *
   * @var \Drupal\layout_builder\SectionStorageInterface
   */
  protected $sectionStorage;

  /**
   * The index of the section containing the block to be cloned.
   *
   * @var int
   */
  protected $delta;

  /**
   * The region inside the section where the block lives.
   *
   * @var string
   */
  protected $region;

  /**
   * The UUID of the block (component) to clone.
   *
   * @var string
   */
  protected $uuid;

  /**
   * Tempstore repository used to persist layout changes during editing.
   *
   * @var \Drupal\layout_builder\LayoutTempstoreRepositoryInterface
   */
  protected $layoutTempstore;

  /**
   * The UUID service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected $uuidService;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  public function __construct(LayoutTempstoreRepositoryInterface $layout_tempstore_repository, UuidInterface $uuid_service, EntityTypeManagerInterface $entity_type_manager) {
    $this->layoutTempstore = $layout_tempstore_repository;
    $this->uuidService = $uuid_service;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   *
   * Factory method used by the service container.
   */
  public static function create(ContainerInterface $container) {
    return new static(
    $container->get('layout_builder.tempstore_repository'),
    $container->get('uuid'),
    $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   *
   * Unique identifier for this form.
   */
  public function getFormId() {
    return 'layout_builder_section_block_duplicate_block_clone';
  }

  /**
   * Page title callback for the block clone form route.
   *
   * @param \Drupal\layout_builder\SectionStorageInterface $section_storage
   *   Section storage containing the block.
   * @param int $delta
   *   Index of the section.
   * @param string $uuid
   *   UUID of the block to clone.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   A translated title with the block label.
   */
  public static function title(SectionStorageInterface $section_storage, $delta, $uuid) {
    $block_label = $section_storage
      ->getSection($delta)
      ->getComponent($uuid)
      ->getPlugin()
      ->label();
    return t('Clone the @block_label block', ['@block_label' => $block_label]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?SectionStorageInterface $section_storage = NULL, $delta = NULL, $region = NULL, $uuid = NULL) {
    $parameters = array_slice(func_get_args(), 2);
    foreach ($parameters as $parameter) {
      if (is_null($parameter)) {
        throw new \InvalidArgumentException('CloneBlockForm requires all parameters.');
      }
    }

    $this->sectionStorage = $section_storage;
    $this->delta = $delta;
    $this->uuid = $uuid;
    $this->region = $region;

    $form['#attributes']['data-layout-builder-target-highlight-id'] =
    $this->blockUpdateHighlightId($uuid);

    $sections = $section_storage->getSections();
    $contexts = $this->getPopulatedContexts($section_storage);
    $region_options = [];

    foreach ($sections as $section_delta => $section) {
      $layout = $section->getLayout($contexts);
      $layout_definition = $layout->getPluginDefinition();

      $section_settings = $section->getLayoutSettings();
      $section_label = !empty($section_settings['label'])
            ? $section_settings['label']
            : $this->t('Section: @delta', ['@delta' => $section_delta + 1]);

      $section_label_text = (string) $section_label;

      foreach ($layout_definition->getRegions() as $region_name => $region_info) {
        $region_options[$section_label_text]["$section_delta:$region_name"] = $this->t(
              '@section, Region: @region',
              ['@section' => $section_label, '@region' => $region_info['label']]
          );
      }
    }

    $selected_region = $this->getSelectedRegion($form_state);
    $selected_delta = $this->getSelectedDelta($form_state);
    $form['region'] = [
      '#type' => 'select',
      '#options' => $region_options,
      '#title' => $this->t('Region'),
      '#default_value' => "$selected_delta:$selected_region",
      '#ajax' => [
        'wrapper' => 'layout-builder-components-table',
        'callback' => '::getComponentsWrapper',
      ],
    ];

    $current_section = $sections[$selected_delta];
    $aria_label = $this->t('Blocks in Section: @section, Region: @region', [
      '@section' => $selected_delta + 1,
      '@region' => $selected_region,
    ]);

    $form['components_wrapper']['components'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Block label'),
        $this->t('Weight'),
      ],
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'table-sort-weight',
        ],
      ],
      '#theme_wrappers' => [
        'container' => [
          '#attributes' => [
            'id' => 'layout-builder-components-table',
            'class' => ['layout-builder-components-table'],
            'aria-label' => $aria_label,
          ],
        ],
      ],
    ];

    /** @var \Drupal\layout_builder\SectionComponent[] $components */
    $components = $current_section->getComponentsByRegion($selected_region);

    if (!isset($components[$uuid])) {
      $components[$uuid] = $sections[$delta]->getComponent($uuid);
    }
    $state_weight_delta = round(count($components) / 2);
    foreach ($components as $component_uuid => $component) {
      $plugin = $component->getPlugin();
      $is_current_block = $component_uuid === $uuid;
      $row_classes = ['draggable', 'layout-builder-components-table__row'];
      $label['#wrapper_attributes']['class'] = ['layout-builder-components-table__block-label'];
      $label['#markup'] = $plugin->label();
      if ($is_current_block) {
        $label['#markup'] .= ' (' . $this->t('current') . ')';
        $label['#wrapper_attributes']['class'][] =
        'layout-builder-components-table__block-label--current';
        $row_classes[] = 'layout-builder-components-table__row--current';
      }
      $form['components_wrapper']['components'][$component_uuid] = [
        '#attributes' => ['class' => $row_classes],
        'label' => $label,
        'weight' => [
          '#type' => 'weight',
          '#default_value' => $component->getWeight(),
          '#title' => $this->t('Weight for @block block', ['@block' => $plugin->label()]),
          '#title_display' => 'invisible',
          '#attributes' => ['class' => ['table-sort-weight']],
          '#delta' => $state_weight_delta,
        ],
      ];
    }

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Clone'),
      '#button_type' => 'primary',
    ];

    $form['#attributes']['data-add-layout-builder-wrapper'] = 'layout-builder--clone-block-active';

    if ($this->isAjax()) {
      $form['actions']['submit']['#ajax']['callback'] = '::ajaxSubmit';
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Get the selected region and section delta from the form state.
    $region = $this->getSelectedRegion($form_state);
    $delta = $this->getSelectedDelta($form_state);

    // Original section and component.
    $original_section = $this->sectionStorage->getSection($this->delta);
    $original_component = $original_section->getComponent($this->uuid);

    // Get the plugin and its configuration for the original component.
    $plugin = $original_component->getPlugin();
    $configuration = $plugin->getConfiguration();

    // Handle inline blocks created via Layout Builder.
    if (!empty($configuration['provider']) && $configuration['provider'] === 'layout_builder') {

      // If the block has a block_uuid, duplicate the block_content entity.
      if (!empty($configuration['block_uuid'])) {
        $original_blocks = $this->entityTypeManager
          ->getStorage('block_content')
          ->loadByProperties(['uuid' => $configuration['block_uuid']]);

        if ($original_block = reset($original_blocks)) {
          $cloned_block = $original_block->createDuplicate();
          $cloned_block->enforceIsNew();
          $cloned_block->set('uuid', $this->uuidService->generate());
          $cloned_block->setNewRevision(TRUE);
          $cloned_block->set('revision_id', NULL);
          $cloned_block->set('id', NULL);
          $cloned_block->save();

          // Update the component configuration to reference the new block.
          $configuration['block_revision_id'] = $cloned_block->getRevisionId();
          $configuration['block_uuid'] = $cloned_block->uuid();
        }
      }

      // If the component has a serialized inline block, duplicate it too.
      if (!empty($configuration['block_serialized'])) {
        $inline_block = unserialize($configuration['block_serialized']);

        if ($inline_block && $inline_block->getEntityTypeId() === 'block_content') {
          $duplicate_block = $inline_block->createDuplicate();
          $duplicate_block->enforceIsNew();
          $duplicate_block->set('uuid', $this->uuidService->generate());
          $duplicate_block->setNewRevision(TRUE);
          $duplicate_block->set('revision_id', NULL);
          $duplicate_block->set('id', NULL);
          $duplicate_block->save();

          // Update configuration to reference the new serialized block.
          $configuration['block_serialized'] = serialize($duplicate_block);
          $configuration['block_revision_id'] = $duplicate_block->getRevisionId();
          $configuration['block_uuid'] = $duplicate_block->uuid();
        }
      }

      // Always regenerate the component's own UUID to prevent conflicts.
      if (isset($configuration['uuid'])) {
        $configuration['uuid'] = $this->uuidService->generate();
      }
    }

    // Create the cloned component with the updated configuration.
    $new_uuid = $this->uuidService->generate();
    $component = new SectionComponent($new_uuid, $plugin, $configuration);
    $component->setRegion($region);
    $component->setWeight($original_component->getWeight());

    $section = $this->sectionStorage->getSection($delta);
    $section->insertComponent(0, $component);

    $components_in_region = $section->getComponentsByRegion($region);
    $components_in_region[$new_uuid] = $component;

    // Apply weights from the draggable table in the form.
    $form_components = $form_state->getValue('components') ?: [];
    foreach ($form_components as $uuid => $component_info) {
      if ($uuid === $this->uuid) {
        // The "current" form row corresponds to the original component.
        // Since we are cloning, the clone should take the selected weight.
        if (isset($components_in_region[$new_uuid])) {
          $components_in_region[$new_uuid]->setWeight($component_info['weight']);
        }
      }
      elseif (isset($components_in_region[$uuid])) {
        $components_in_region[$uuid]->setWeight($component_info['weight']);
      }
    }

    // Save the updated section storage into tempstore.
    $this->layoutTempstore->set($this->sectionStorage);
    $this->messenger()->addStatus($this->t('Block cloned successfully.'));
  }

  /**
   * Ajax callback for the region select element.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The components wrapper render array.
   */
  public function getComponentsWrapper(array $form, FormStateInterface $form_state) {
    return $form['components_wrapper'];
  }

  /**
   * {@inheritdoc}
   */
  protected function getSelectedRegion(FormStateInterface $form_state) {
    if ($form_state->hasValue('region')) {
      return explode(':', $form_state->getValue('region'), 2)[1];
    }
    return $this->region;
  }

  /**
   * Gets the selected region.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return string
   *   The current region name.
   */
  protected function getSelectedDelta(FormStateInterface $form_state) {
    if ($form_state->hasValue('region')) {
      return (int) explode(':', $form_state->getValue('region'))[0];
    }
    return (int) $this->delta;
  }

  /**
   * Callback for a successful AJAX submit.
   */
  protected function successfulAjaxSubmit(array $form, FormStateInterface $form_state) {
    return $this->rebuildAndClose($this->sectionStorage);
  }

}
