<?php

namespace Drupal\layout_builder_section_block_duplicate;

use Drupal\Core\Url;
use Drupal\Core\Render\Element;
use Drupal\Core\Security\TrustedCallbackInterface;

/**
 * Alters the Layout Builder render array.
 *
 * Adds custom action links to sections.
 */
class LayoutBuilderAlter implements TrustedCallbackInterface {

  /**
   * Adds an extra link below the "Configure" option to clone sections.
   *
   * @param array $element
   *   The Layout Builder render array.
   *
   * @return array
   *   The modified render array with the additional link.
   */
  public static function addCustomLink(array $element): array {
    if (!empty($element['layout_builder']) && is_array($element['layout_builder'])) {
      foreach (Element::children($element['layout_builder']) as $child) {
        $section = &$element['layout_builder'][$child];

        if (isset($section['configure'])) {
          $params = $section['configure']['#url']->getRouteParameters();
          $delta = $params['delta'];
          $storage_type = $params['section_storage_type'];
          $storage_id = $params['section_storage'];

          $class = self::isGinLbEnabled()
            ? 'layout-builder__link--clone-gin'
            : 'layout-builder__link--clone-claro';

          $section['custom_action'] = [
            '#type' => 'link',
            '#title' => t('Clone section @num', ['@num' => $delta + 1]),
            '#url' => Url::fromRoute('layout_builder_section_block_duplicate.clone_section', [
              'section_storage_type' => $storage_type,
              'section_storage' => $storage_id,
              'delta' => $delta,
            ]),
            '#attributes' => [
              'class' => ['use-ajax', 'layout-builder__link', $class],
              'data-dialog-type' => 'dialog',
              'data-dialog-renderer' => 'off_canvas',
            ],
            '#weight' => -1,
          ];
        }
      }
    }

    return $element;
  }

  /**
   * Declares trusted callbacks for Drupal.
   *
   * Methods listed here can be safely called to avoid security issues.
   *
   * @return array
   *   A list of trusted method names.
   */
  public static function trustedCallbacks(): array {
    return ['addCustomLink'];
  }

  /**
   * Checks if the Gin Layout Builder (gin_lb) module is enabled.
   *
   * @return bool
   *   TRUE if gin_lb is enabled, FALSE otherwise.
   */
  protected static function isGinLbEnabled(): bool {
    return \Drupal::service('module_handler')->moduleExists('gin_lb');
  }

}
