(($, Drupal, drupalSettings, once) => {
  /**
   * Get context information about an element.
   * @param {Element} el The element to get context for.
   * @return {Object} Context information about the element.
   */
  function getContext(el) {
    return {
      parent_type: el.closest('[data-lp-component-type]')
        ? el
            .closest('[data-lp-component-type]')
            .getAttribute('data-lp-component-type')
        : null,
      region:
        el.closest('[data-region]')?.getAttribute('data-region') || '_root',
      layout: el.closest('[data-layout]')
        ? el.closest('[data-layout]').getAttribute('data-layout')
        : null,
      sibling_type: el.previousElementSibling
        ? el.previousElementSibling.getAttribute('data-lp-component-type')
        : null,
      field_name: el.closest('[data-lp-reference-field]')
        ? el
            .closest('[data-lp-reference-field]')
            .getAttribute('data-lp-reference-field')
        : null,
      entity_type: el.closest('[data-lp-entity-type]')
        ? el
            .closest('[data-lp-entity-type]')
            .getAttribute('data-lp-entity-type')
        : null,
      entity_bundle: el.closest('[data-lp-entity-bundle]')
        ? el
            .closest('[data-lp-entity-bundle]')
            .getAttribute('data-lp-entity-bundle')
        : null,
    };
  }

  /**
   * Get counts of components within a target container.
   * @param {Element} source The element being moved.
   * @param {Element} target The target container.
   * @return {Object} An object of component type (bundle) to count, including a _total key for total count.
   */
  function getCounts(source, target) {
    const srcUuid = source.closest('[data-uuid]')?.getAttribute('data-uuid');
    const nestedComponents = Array.from(
      target.querySelectorAll(`[data-uuid]:not([data-uuid="${srcUuid}"])`),
    ).filter((child) => {
      return (
        child.parentNode.closest('[data-uuid]') ===
        target.closest('[data-uuid]')
      );
    });
    const counts = {
      _total: nestedComponents.length,
    };
    nestedComponents.forEach((child) => {
      const componentType = child.getAttribute('data-lp-component-type');
      if (componentType) {
        counts[componentType] = (counts[componentType] || 0) + 1;
      }
    });
    return counts;
  }

  /**
   * Get the maximum allowed items from a list of restrictions.
   * @param {Array} restrictions The list of restrictions to check.
   * @param {string} type The component type (bundle) to check.
   * @return {Object} An object with _total and type keys for maximum allowed items.
   */
  function getMaxItems(restrictions, type) {
    const maxItems = {
      _total: null,
      [type]: null,
    };
    if (restrictions.length > 0) {
      restrictions.forEach((restriction) => {
        if (restriction.max_items) {
          // Get minimum _total value
          if (restriction.max_items._total !== undefined) {
            maxItems._total =
              maxItems._total === null
                ? restriction.max_items._total
                : Math.min(maxItems._total, restriction.max_items._total);
          }

          // Get minimum value for current component type
          if (restriction.max_items[type] !== undefined) {
            maxItems[type] =
              maxItems[type] === null
                ? restriction.max_items[type]
                : Math.min(maxItems[type], restriction.max_items[type]);
          }
        }
      });
    }
    return maxItems;
  }

  /**
   * Returns true if all keys in obj1 exist in obj2 with the same value.
   * Handles negation values (e.g. "!value").
   * @param {Object} obj1 The object to check.
   * @param {Object} obj2 The object to check against.
   * @return {boolean} True if all keys in obj1 exist in obj2 with the same value.
   */
  function matchesContext(obj1, obj2) {
    return Object.entries(obj1).every(([key, value]) => {
      // Skip if key doesn't exist in target context.
      if (!obj2[key]) {
        return false;
      }

      // Handle negation cases.
      if (value.startsWith('!')) {
        return obj2[key] !== value.substring(1);
      }

      // Direct value comparison.
      return obj2[key] === value;
    });
  }

  /**
   * Filters a list of restrictions to only those that apply to the current context.
   * @param {Array} restrictions The list of restrictions to filter.
   * @param {Object} targetContext The context to filter against.
   * @return {Array} The list of restrictions that apply to the current context.
   */
  function applicableRestrictions(restrictions, targetContext) {
    return restrictions.filter((restriction) => {
      const contexts = Array.isArray(restriction.context)
        ? restriction.context
        : [restriction.context];
      return contexts.some((context) => matchesContext(context, targetContext));
    });
  }

  /**
   * Returns the applicable move errors for a given.
   * @param {Object} allRestrictions The Drupal settings object.
   * @param {Element} el The element being moved.
   * @param {Element} target The target container.
   * @param {Element} source The source container.
   * @param {Element} sibling The sibling element.
   * @return {string|undefined} The error message, or undefined if no error.
   */
  function applyRestrictions(allRestrictions, el, target) {
    // The context that element is being moved into.
    const moveContext = getContext(target);

    // Filters the list of restrictions to only those that apply to the current context.
    const appliedRestrictions = applicableRestrictions(
      allRestrictions,
      moveContext,
    );

    const type = el.getAttribute('data-lp-component-type') || [];

    // Handle include and exclude lists from restrictions that apply.
    if (appliedRestrictions.length > 0) {
      const exclude = appliedRestrictions.reduce(
        (excludedByRule, restriction) => [
          ...excludedByRule,
          ...(restriction.exclude_components || []),
        ],
        [],
      );
      const include = appliedRestrictions.reduce(
        (includedByRule, restriction) => [
          ...includedByRule,
          ...(restriction.components || []),
        ],
        [],
      );
      // Compare the type of the element to the include/exclude lists.
      if (exclude.length > 0 && exclude.indexOf(type) !== -1) {
        return 'This component cannot be moved here.';
      }
      if (include.length > 0 && include.indexOf(type) === -1) {
        return 'This component cannot be moved here.';
      }
    }

    // Handle max items restrictions.
    const maxItems = getMaxItems(appliedRestrictions, type);
    const counts = getCounts(el, target);

    // Check max items for the region.
    if (maxItems._total !== null && counts._total >= maxItems._total) {
      return `Only ${maxItems._total} items are allowed in this area.`;
    }

    // Check max items for the component type.
    if (maxItems[type] !== null && counts[type] >= maxItems[type]) {
      return `Only ${maxItems[type]} of this type of component are allowed in this area.`;
    }
  }

  Drupal.behaviors.layoutParagraphsRestrictions = {
    attach: function attach(context, settings) {
      $(once('lpb-restrictions', '[data-lpb-id]')).each((i, e) => {
        const lpbId = e.getAttribute('data-lpb-id');
        if (
          typeof settings.lpBuilder.restrictions !== 'undefined' &&
          typeof settings.lpBuilder.restrictions[lpbId] !== 'undefined'
        ) {
          // const allRestrictions = settings.lpBuilder.restrictions[lpbId];
          Drupal.registerLpbMoveError((_lpbSettings, el, target) =>
            applyRestrictions(
              settings.lpBuilder.restrictions[lpbId],
              el,
              target,
            ),
          );
        }
      });
    },
  };
})(jQuery, Drupal, drupalSettings, once);
