<?php

namespace Drupal\layout_paragraphs_restrictions\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\Messenger;
use Drupal\layout_paragraphs\Event\LayoutParagraphsAllowedTypesEvent;
use Drupal\layout_paragraphs\Event\LayoutParagraphsDuplicateEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Provides Layout Paragraphs Restrictions.
 */
class LayoutParagraphsRestrictions implements EventSubscriberInterface {

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\Messenger
   */
  protected $messenger;

  /**
   * An array of restriction rules.
   *
   * @var array
   */
  protected $restrictions;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructor.
   *
   * We use dependency injection get Messenger.
   *
   * @param \Drupal\Core\Messenger\Messenger $messenger
   *   The messenger service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   */
  public function __construct(Messenger $messenger, ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory) {
    $this->messenger = $messenger;
    $this->restrictions = $config_factory
      ->get('layout_paragraphs_restrictions.settings')
      ->get('restrictions');
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritDoc}
   */
  public static function getSubscribedEvents() {
    return [
      LayoutParagraphsAllowedTypesEvent::EVENT_NAME => [
        'typeRestrictions',
        -100,
      ],
      LayoutParagraphsDuplicateEvent::EVENT_NAME => [
        'duplicateRestrictions',
        -100,
      ],
    ];
  }

  /**
   * Restricts available types based on settings in layout.
   *
   * The current context will have the following possible key/value pairs:
   *
   * - parent_uuid: The UUID of the parent component.
   * - parent_type: The bundle of the parent component.
   * - sibling_uuid: The UUID of the sibling component.
   * - sibling_type: The bundle of the sibling component.
   * - region: The region name (_root if no region is set).
   * - layout: The layout plugin ID.
   * - placement: The placement of the component (before or after).
   * - field_name: The field name.
   *
   * @param \Drupal\layout_paragraphs\Event\LayoutParagraphsAllowedTypesEvent $event
   *   The allowed types event.
   */
  public function typeRestrictions(LayoutParagraphsAllowedTypesEvent $event) {
    $context = $event->getContext();

    $componentCounts = $this->getComponentCounts(
      $context['parent_uuid'] ?? '',
      $context['region'] ?? '',
      $event->getLayout(),
    );

    if ($context['parent_uuid']) {
      $parent_paragraph = $event->getLayout()->getComponentByUuid($event->getParentUuid())->getEntity();
      $section = $event->getLayout()->getLayoutSection($parent_paragraph);
      $layout_plugin_id = $section->getLayoutId();
      $context['parent_type'] = $parent_paragraph->bundle();
      $context['layout'] = $layout_plugin_id;
    }
    if ($context['sibling_uuid']) {
      $sibling_paragraph = $event->getLayout()->getComponentByUuid($context['sibling_uuid'])->getEntity();
      $context['sibling_type'] = $sibling_paragraph->bundle();
    }
    if (empty($context['region'])) {
      $context['region'] = '_root';
    }

    // @todo These contexts are not returned by Layout Paragraphs.
    $paragraphsReferenceField = $event->getLayout()->getParagraphsReferenceField();
    $parentEntity = $paragraphsReferenceField->getEntity();

    if (empty($context['field_name'])) {
      $context['field_name'] = $paragraphsReferenceField->getName();
    }
    if (empty($context['entity_type'])) {
      $context['entity_type'] = $parentEntity->getEntityTypeId();
    }
    if (empty($context['entity_bundle'])) {
      $context['entity_bundle'] = $parentEntity->bundle();
    }

    foreach ($this->restrictions as $restriction) {
      $include = [];
      $exclude = [];

      if ($this->contextMatches($restriction['context'], $context)) {
        if (!empty($restriction['components'])) {
          $include = array_merge($include, array_fill_keys($restriction['components'], TRUE));
        }
        if (!empty($restriction['exclude_components'])) {
          $exclude = array_merge($exclude, array_fill_keys($restriction['exclude_components'], TRUE));
        }

        if ($include) {
          $event->setTypes(array_intersect_key($event->getTypes(), $include));
        }
        if ($exclude) {
          $event->setTypes(array_diff_key($event->getTypes(), $exclude));
        }
        if (!empty($restriction['max_items'])) {
          if (!empty($restriction['max_items']['_total']) && $restriction['max_items']['_total'] <= $componentCounts['_total']) {
            // If the total number of components has reached the max, no more
            // components can be added.
            $event->setTypes([]);
          }
          foreach ($event->getTypes() as $type => $label) {
            // If max counts are specified, enforce them.
            if (!empty($restriction['max_items'][$type]) && isset($componentCounts[$type]) && $restriction['max_items'][$type] <= $componentCounts[$type]) {
              // If the number of components of this type has reached the max,
              // remove it from the allowed types.
              $event->setTypes(array_diff_key($event->getTypes(), [$type => TRUE]));
            }
          }
        }
      }
    }
  }

  /**
   * Tests if a restriction context matches the current context.
   *
   * @param array $restriction_context
   *   The restriction context configuration.
   * @param array $current_context
   *   The current context to test against.
   *
   * @return bool
   *   TRUE if the context matches, FALSE otherwise.
   */
  protected function contextMatches(array $restriction_context, array $current_context) {
    $tests = [];

    // If context is a numerically indexed array, treat each item as a
    // separate test context. Otherwise treat it as a single test context.
    if (is_array($restriction_context) && array_keys($restriction_context) === range(0, count($restriction_context) - 1)) {
      $tests = array_merge($tests, $restriction_context);
    }
    else {
      $tests[] = $restriction_context;
    }

    foreach ($tests as $runtest) {
      $total_tests = 0;
      $matched_tests = 0;

      foreach ($runtest as $restriction_context_key => $restriction_context_value) {
        $total_tests++;

        if (str_starts_with($restriction_context_value, '!')) {
          $restriction_context_value = substr($restriction_context_value, 1);
          $test = isset($current_context[$restriction_context_key])
            && $current_context[$restriction_context_key] !== $restriction_context_value;
        }
        else {
          $test = isset($current_context[$restriction_context_key])
            && $current_context[$restriction_context_key] === $restriction_context_value;
        }

        if ($test === TRUE) {
          $matched_tests++;
        }
      }

      // If all tests matched, return true.
      if ($matched_tests === $total_tests) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Counts components of each type within a given parent and region.
   *
   * @param string $parent_uuid
   *   The parent UUID.
   * @param string $region
   *   The region name.
   * @param \Drupal\layout_paragraphs\LayoutParagraphsLayout $layout
   *   The layout object.
   *
   * @return array
   *   An associative array of component type (bundle) to count.
   */
  protected function getComponentCounts($parent_uuid, $region, $layout) {
    $counts = [
      '_total' => 0,
    ];

    // Get the total number of components in the layout region,
    // as well as the total per component type.
    if ($region && $parent_uuid) {
      $parent_component = $layout->getComponentByUuid($parent_uuid);
      if ($parent_component) {
        $section = $layout->getLayoutSection($parent_component->getEntity());
        $components = $section->getComponentsForRegion($region);
      }
    }
    else {
      $components = $layout->getRootComponents();
    }

    $counts['_total'] = count($components);
    foreach ($components as $component) {
      $bundle = $component->getEntity()->bundle();
      if (!isset($counts[$bundle])) {
        $counts[$bundle] = 0;
      }
      $counts[$bundle]++;
    }
    return $counts;
  }

  /**
   * Prevents duplication when max_items restrictions would be violated.
   *
   * @param \Drupal\layout_paragraphs\Event\LayoutParagraphsDuplicateEvent $event
   *   The duplication event.
   */
  public function duplicateRestrictions(LayoutParagraphsDuplicateEvent $event) {
    $layout = $event->getLayout();
    $duplicate_component = $event->getDuplicateComponent();
    $duplicate_entity = $duplicate_component->getEntity();

    // Build context for the duplicated component.
    $context = [
      'region' => '_root',
    ];

    // Check if the component is within a layout section.
    $parent_uuid = $duplicate_component->getParentUuid();
    if ($parent_uuid) {
      $parent_component = $layout->getComponentByUuid($parent_uuid);
      $parent_entity = $parent_component->getEntity();
      $context['parent_uuid'] = $parent_uuid;
      $context['parent_type'] = $parent_entity->bundle();

      $section = $layout->getLayoutSection($parent_entity);
      $context['layout'] = $section->getLayoutId();
      $context['region'] = $duplicate_component->getRegion();
    }

    // Get component counts for the target location.
    $componentCounts = $this->getComponentCounts(
      $context['parent_uuid'] ?? '',
      $context['region'] ?? '',
      $layout,
    );

    // Add the entity type and bundle context.
    $paragraphsReferenceField = $layout->getParagraphsReferenceField();
    $parentEntity = $paragraphsReferenceField->getEntity();
    $context['field_name'] = $paragraphsReferenceField->getName();
    $context['entity_type'] = $parentEntity->getEntityTypeId();
    $context['entity_bundle'] = $parentEntity->bundle();

    // Check each restriction rule.
    foreach ($this->restrictions as $restriction) {
      if (empty($restriction['max_items'])) {
        continue;
      }

      // If all tests matched, check max_items restrictions.
      if ($this->contextMatches($restriction['context'], $context)) {
        $duplicate_bundle = $duplicate_entity->bundle();

        // Check total max items.
        if (!empty($restriction['max_items']['_total']) && $restriction['max_items']['_total'] < $componentCounts['_total']) {
          $event->preventDuplication('Cannot duplicate: Maximum total items (' . $restriction['max_items']['_total'] . ') reached for this location.');
          return;
        }

        // Check type-specific max items.
        if (!empty($restriction['max_items'][$duplicate_bundle])) {
          $current_count = $componentCounts[$duplicate_bundle] ?? 0;
          if ($restriction['max_items'][$duplicate_bundle] < $current_count) {
            $event->preventDuplication('Cannot duplicate: Maximum items of type "' . $duplicate_bundle . '" (' . $restriction['max_items'][$duplicate_bundle] . ') reached for this location.');
            return;
          }
        }
      }
    }
  }

}
