<?php

namespace Drupal\lazy_iframe;

use Drupal\Core\Render\Element;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for processing iframes to add lazy loading.
 */
class LazyIframeProcessor {

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a LazyIframeProcessor object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory) {
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Process content to add lazy loading to iframes.
   *
   * @param string $content
   *   The HTML content to process.
   *
   * @return string
   *   The processed content with lazy loading attributes.
   */
  public function processContent(string $content): string {
    if (empty(trim($content))) {
      return $content;
    }

    if (strpos($content, '<iframe') === FALSE) {
      return $content;
    }

    try {
      return $this->processHtmlContent($content);
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('lazy_iframe')->error(
        'Error processing iframe content: @message',
        ['@message' => $e->getMessage()]
      );
      return $content;
    }
  }

  /**
   * Process render element to add lazy loading to iframes.
   *
   * @param array $element
   *   The render element to process.
   *
   * @return array
   *   The processed render element.
   */
  public function processElement(array $element): array {
    if (isset($element['#markup']) && strpos($element['#markup'], '<iframe') !== FALSE) {
      if (!$this->hasSkippableIframes($element['#markup'])) {
        $element['#markup'] = $this->processContent($element['#markup']);
      }
    }

    foreach (Element::children($element) as $key) {
      $element[$key] = $this->processElement($element[$key]);
    }

    return $element;
  }

  /**
   * Check if content has iframes that should be skipped.
   *
   * @param string $content
   *   The HTML content to check.
   *
   * @return bool
   *   TRUE if content has skippable iframes, FALSE otherwise.
   */
  protected function hasSkippableIframes(string $content): bool {
    $patterns = [
      '/<iframe[^>]*(?!.*src\s*=\s*["\'][^"\']+["\'])[^>]*>/i',
      '/<iframe[^>]*src\s*=\s*["\'][\s]*["\'][^>]*>/i',
      '/<iframe[^>]*src\s*=\s*["\']#["\'][^>]*>/i',
    ];

    foreach ($patterns as $pattern) {
      if (preg_match($pattern, $content)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Process HTML content using DOMDocument.
   *
   * @param string $content
   *   The HTML content to process.
   *
   * @return string
   *   The processed content.
   */
  protected function processHtmlContent(string $content): string {
    $previousUseErrors = libxml_use_internal_errors(TRUE);
    libxml_clear_errors();

    try {
      $doc = new \DOMDocument('1.0', 'UTF-8');

      $success = $doc->loadHTML(
        '<?xml encoding="UTF-8"><div>' . $content . '</div>',
        LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD
      );

      if (!$success) {
        throw new \RuntimeException('Failed to parse HTML content');
      }

      $iframes = $doc->getElementsByTagName('iframe');

      foreach ($iframes as $iframe) {
        if (!$iframe->hasAttribute('loading')) {
          $iframe->setAttribute('loading', 'lazy');
        }
      }

      $body = $doc->getElementsByTagName('div')->item(0);
      $processedContent = '';

      if ($body && $body->childNodes) {
        foreach ($body->childNodes as $child) {
          $processedContent .= $doc->saveHTML($child);
        }
      }

      return $processedContent;
    }
    finally {
      libxml_use_internal_errors($previousUseErrors);
    }
  }

}
