<?php

namespace Drupal\lb_plus;

use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;

/**
 * Compiler pass to add workspace awareness to Layout Builder tempstore.
 *
 * This runs AFTER all ServiceProviders (including page_manager_ui's) to ensure
 * we properly decorate the layout_builder.tempstore_repository service
 * regardless of ServiceProvider execution order.
 */
class WorkspaceAwareTempstorePass implements CompilerPassInterface {

  /**
   * {@inheritdoc}
   */
  public function process(ContainerBuilder $container) {
    // Only proceed if workspaces module is enabled.
    if (!$container->hasDefinition('workspaces.manager')) {
      return;
    }

    if (!$container->hasDefinition('layout_builder.tempstore_repository')) {
      return;
    }

    $definition = $container->getDefinition('layout_builder.tempstore_repository');
    $tempstore_factory = $definition->getArgument(0);

    // Detect which class is currently set (could be core's or page_manager_ui's).
    $current_class = $definition->getClass();

    if ($current_class === 'Drupal\page_manager_ui\LayoutBuilderTempstoreRepository') {
      // page_manager_ui has decorated the service, use our page-manager-aware version.
      $definition->setClass(WorkspaceAwarePageManagerLayoutTempstoreRepository::class);
    }
    else {
      // Use standard workspace-aware version for core's LayoutTempstoreRepository.
      $definition->setClass(WorkspaceAwareLayoutTempstoreRepository::class);
    }

    // Set arguments explicitly (both classes have same constructor signature).
    $definition->setArguments([
      $tempstore_factory,
      new Reference('workspaces.manager'),
    ]);

    $container->setDefinition('layout_builder.tempstore_repository', $definition);
  }

}
