<?php

namespace Drupal\lb_plus;

use Drupal\Core\Entity\TranslatableInterface;
use Drupal\layout_builder\SectionStorageInterface;

/**
 * Provides workspace-aware tempstore key generation.
 *
 * This trait provides helper methods for adding workspace ID and langcode
 * to tempstore keys.
 *
 * Classes using this trait must provide:
 * - $this->workspaceManager (WorkspaceManagerInterface|null)
 */
trait WorkspaceAwareTempstoreTrait {

  /**
   * {@inheritdoc}
   */
  protected function getKey(SectionStorageInterface $section_storage) {
    // Get the base key from parent class.
    $key = parent::getKey($section_storage);

    // Ensure langcode is always present in the key for consistency.
    $key = $this->ensureLangcodeInKey($key, $section_storage);

    // Append workspace ID if workspaces module is enabled.
    if ($this->workspaceManager !== NULL) {
      if ($this->workspaceManager->hasActiveWorkspace()) {
        $key .= '.' . $this->workspaceManager->getActiveWorkspace()->id();
      }
      else {
        // Use 'live' for the default/live workspace.
        $key .= '.live';
      }
    }

    return $key;
  }

  /**
   * Ensures langcode is present in the tempstore key.
   *
   * Core's OverridesSectionStorage includes langcode for translatable entities,
   * but we want to ensure langcode is always present for consistency.
   *
   * @param string $key
   *   The current tempstore key.
   * @param \Drupal\layout_builder\SectionStorageInterface $section_storage
   *   The section storage.
   *
   * @return string
   *   The key with langcode ensured.
   */
  protected function ensureLangcodeInKey($key, SectionStorageInterface $section_storage) {
    // Check if langcode might already be in the key.
    // Expected formats:
    // - OverridesSectionStorage: node.1.full.en (has langcode)
    // - DefaultsSectionStorage: node.landing_page.full (no langcode)
    $parts = explode('.', $key);

    // If we have 4+ parts for overrides, langcode is likely already present.
    // Format: entity_type.id.view_mode.langcode
    if (count($parts) >= 4) {
      return $key;
    }

    // For DefaultsSectionStorage or cases where langcode is missing,
    // try to get langcode from entity context if available.
    try {
      // Check if the method exists before calling it (not all section storage
      // types implement the full context API).
      if (method_exists($section_storage, 'hasContextValue') && $section_storage->hasContextValue('entity')) {
        $entity = $section_storage->getContextValue('entity');
        if ($entity instanceof TranslatableInterface) {
          $langcode = $entity->language()->getId();
        }
        else {
          // Use 'und' (undefined) for non-translatable entities.
          $langcode = 'und';
        }
        return $key . '.' . $langcode;
      }
    }
    catch (\Exception $e) {
      // Context not available or error accessing it.
      // This can happen with DefaultsSectionStorage which doesn't have
      // entity context.
    }

    // Fallback: append 'und' if we can't determine langcode.
    // This ensures consistent key format even for config-based section storage.
    return $key . '.und';
  }

}
