<?php

namespace Drupal\lb_theme_switcher\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure the name of the theme to switch to.
 */
class ThemeSwitcherSettingsForm extends ConfigFormBase {

  /**
   * Theme Handler service.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected ThemeHandlerInterface $themeHandler;

  /**
   * Constructs a ThemeSwitcherForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Extension\ThemeHandlerInterface $theme_handler
   *   Theme handler service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *    The typed config manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    protected TypedConfigManagerInterface $typedConfigManager,
    ThemeHandlerInterface $theme_handler
  ) {
    parent::__construct($config_factory, $typedConfigManager);

    $this->themeHandler = $theme_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('theme_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'lb_theme_switcher_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['lb_theme_switcher.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Get the list of installed themes.
    $themes = $this->themeHandler->listInfo();
    $theme_options = [];

    // Populate the options array with enabled themes.
    foreach ($themes as $theme_name => $theme_info) {
      if ($theme_info->status) {
        $theme_options[$theme_name] = $theme_info->info['name'];
      }
    }

    // Get the currently configured theme.
    $config = $this->config('lb_theme_switcher.settings');
    $default_theme = $config->get('theme_name');

    $form['theme_name'] = [
      '#type' => 'select',
      '#title' => $this->t('Theme Name'),
      '#description' => $this->t('Select the theme you want to switch to on Layout Builder pages.'),
      '#options' => $theme_options,
      '#default_value' => $default_theme,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $theme_name = $form_state->getValue('theme_name');
    $themes = $this->themeHandler->listInfo();

    if (!isset($themes[$theme_name]) || !$themes[$theme_name]->status) {
      $form_state->setErrorByName('theme_name', $this->t('The selected theme is not valid.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Save the selected theme into the configuration.
    $this->configFactory->getEditable('lb_theme_switcher.settings')
      ->set('theme_name', $form_state->getValue('theme_name'))
      ->save();

    $this->messenger()
      ->addMessage($this->t('The Layout Builder pages will now have %theme theme.', ['%theme' => $form_state->getValue('theme_name')]));

    parent::submitForm($form, $form_state);
  }

}
