<?php

namespace Drupal\lb_theme_switcher\Theme;

use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Theme\ThemeNegotiatorInterface;
use Drupal\layout_builder\Plugin\SectionStorage\OverridesSectionStorage;
use Drupal\node\NodeInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Switch to a Layout Builder-compatible theme when on LB pages.
 */
class LayoutBuilderThemeNegotiator implements ThemeNegotiatorInterface {

  /**
   * AdminContext service.
   *
   * @var \Drupal\Core\Routing\AdminContext
   */
  protected AdminContext $adminContext;

  /**
   * Theme switcher configuration object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * Request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The CSRF token generator.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected CsrfTokenGenerator $csrfGenerator;

  /**
   * LayoutBuilderThemeNegotiator constructor.
   *
   * @param \Drupal\Core\Routing\AdminContext $admin_context
   *   The AdminContext service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $token_generator
   *   The CSRF token generator.
   */
  public function __construct(
    AdminContext $admin_context,
    ConfigFactoryInterface $config_factory,
    RequestStack $request_stack,
    CsrfTokenGenerator $token_generator
  ) {
    $this->adminContext = $admin_context;
    $this->config = $config_factory->get('lb_theme_switcher.settings');
    $this->requestStack = $request_stack;
    $this->csrfGenerator = $token_generator;
  }

  /**
   * {@inheritdoc}
   */
  public function applies(RouteMatchInterface $route_match) {
    if ($this->adminContext->isAdminRoute()) {
      return FALSE;
    }

    // Check if the request is an AJAX request.
    $ajax_page_state = $this->requestStack->getCurrentRequest()
      ->get('ajax_page_state');
    if (!empty($ajax_page_state)) {
      return !empty($ajax_page_state['theme']) && isset($ajax_page_state['theme_token']);
    }

    $node = $route_match->getParameter('node') ?? FALSE;
    $lb_used = FALSE;
    if ($node instanceof NodeInterface) {
      if ($node->hasField('field_use_layout_builder')) {
        $lb_used = (bool) $node->field_use_layout_builder->value;
      }
      elseif ($node->hasField(OverridesSectionStorage::FIELD_NAME)) {
        $lb_used = TRUE;
      }
    }

    if (
      (
        $node instanceof NodeInterface
        && $route_match->getRouteName() == 'entity.node.canonical'
        && $lb_used
      )
      || $route_match->getRouteName() == 'layout_builder.overrides.node.view'
      || $route_match->getRouteName() == 'layout_builder.defaults.node.view'
    ) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function determineActiveTheme(RouteMatchInterface $route_match) {
    $ajax_page_state = $this->requestStack->getCurrentRequest()
      ->get('ajax_page_state');

    if (!empty($ajax_page_state)) {
      $theme = $ajax_page_state['theme'];
      $token = $ajax_page_state['theme_token'];
      if (!$token || $this->csrfGenerator->validate($token, $theme)) {
        return $theme;
      }
    }

    return $this->config->get('theme_name');
  }

}
