<?php

namespace Drupal\lb_theme_switcher\Drush\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\layout_builder\Entity\LayoutBuilderEntityViewDisplay;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Drush\Exceptions\UserAbortException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A Drush commandfile.
 */
final class LbThemeSwitcherCommands extends DrushCommands {

  /**
   * Constructs a LbThemeSwitcherCommands object.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Reset layout builder header and footer for a given bundle.
   */
  #[CLI\Command(name: 'lb_theme_switcher:reset-lb-header-footer', aliases: ['lbreset'])]
  #[CLI\Argument(name: 'bundle', description: 'Bundle to reset header and footer for.')]
  #[CLI\Argument(name: 'mode', description: 'The view mode of the bundle that contains LB. Defaults to "default".')]
  #[CLI\Option(name: 'dry-run', description: 'If set, do not save changes; show what would change.')]
  #[CLI\Usage(name: 'lb_theme_switcher:reset-lb-header-footer landing_page_lb', description: 'Reset Landing Page LB header and footer.')]
  #[CLI\Usage(name: 'lb_theme_switcher:reset-lb-header-footer branch full', description: 'Reset Branch header and footer. Location content types use "full" view mode.')]
  #[CLI\Usage(name: 'lbreset all', description: 'Use the alias and reset LB header and footer for all Layout Builder-enabled bundle/displays.')]

  public function resetLBHeaderFooter($bundle, $mode = 'default', $dryRun = FALSE): void {
    if (!$bundle) {
      return;
    }

    // If 'all' is passed, iterate all node entity view displays and process
    // those that are Layout Builder displays.
    if ($bundle === 'all') {
      // Confirm before proceeding.
      $really = $this->io()->confirm('Are you sure you want to reset the layout builder header and footer for ALL content types and display modes?');
      if (!$really) {
        throw new UserAbortException("Command cancelled.");
      }

      // Load all entity view displays for nodes.
      $displays = $this->entityTypeManager
        ->getStorage('entity_view_display')
        ->loadByProperties(['targetEntityType' => 'node']);

      if (empty($displays)) {
        $this->io()->writeln('No node view displays found.');
        return;
      }

      // First pass: collect only displays that will actually be processed.
      $to_process = [];
      foreach ($displays as $display) {
        // Only consider layout builder displays.
        if (!($display instanceof LayoutBuilderEntityViewDisplay)) {
          continue;
        }

        // Derive bundle and mode from the display id "node.<bundle>.<mode>".
        $display_id = $display->id();
        $parts = explode('.', $display_id);
        if (count($parts) < 3 || $parts[0] !== 'node') {
          continue;
        }
        $bundle_name = $parts[1];
        $mode_name = $parts[2];

        // Skip displays whose template doesn't contain header/footer.
        $template_sections = $display->getSections();
        [$h, $f] = $this->getHeaderFooter($template_sections);
        if (!$h || !$f) {
          continue;
        }

        // Skip bundles with no nodes.
        $nids = \Drupal::entityQuery('node')
          ->condition('type', $bundle_name)
          ->accessCheck(FALSE)
          ->execute();
        if (empty($nids)) {
          continue;
        }

        $to_process[] = [
          'bundle' => $bundle_name,
          'mode' => $mode_name,
          'display' => $display,
          'nids_count' => count($nids),
        ];
      }

      if (empty($to_process)) {
        $this->io()->writeln('No Layout Builder displays with header/footer and nodes found to process.');
        return;
      }

      // Summarize what will be processed (only displays that passed the checks).
      $this->io()->writeln('The following bundle.display modes will be processed:');
      foreach ($to_process as $item) {
        $this->io()->writeln(' - ' . $item['bundle'] . '.' . $item['mode'] . ' (nodes: ' . $item['nids_count'] . ')');
      }

      // Second pass: actually perform the processing and collect results where changes occurred.
      $summary = [];
      foreach ($to_process as $item) {
        [$total, $updated] = $this->processBundleMode($item['bundle'], $item['mode'], $item['display'], $dryRun);
        if ($updated > 0) {
          $summary[] = $item['bundle'] . '.' . $item['mode'] . ' - Found nodes: ' . $total . ', Updated nodes: ' . $updated;
        }
      }

      // Print post-processing summary only for bundles where nodes were changed.
      if (empty($summary)) {
        $this->io()->writeln('Processing complete — no nodes were changed.');
      }
      else {
        $this->io()->writeln('Processing complete — changes applied for:');
        foreach ($summary as $line) {
          $this->io()->writeln($line);
        }
      }

      return;
    }

    // Single bundle/mode path: confirm and process.
    // Confirm before proceeding for single-bundle runs.
    $really = $this->io()->confirm('Are you sure you want to reset the layout builder header and footer for ' . $bundle . ' nodes to the template from the ' . $mode . ' display?');
    if (!$really) {
      throw new UserAbortException("Command cancelled.");
    }

    // Load the default layout for the requested bundle/mode.
    /** @var LayoutBuilderEntityViewDisplay $default_layout */
    $default_layout = $this->entityTypeManager
      ->getStorage('entity_view_display')
      ->load('node.' . $bundle . '.' . $mode);
    if (!($default_layout instanceof LayoutBuilderEntityViewDisplay)) {
      $this->logger()->error('Default layout not found.');
      return;
    }

    [$total_nodes, $updated_nodes] = $this->processBundleMode($bundle, $mode, $default_layout, $dryRun);

    $this->io()->writeln($bundle . ' update - Found nodes: ' . $total_nodes . ', Updated nodes: ' . $updated_nodes);
  }

  /**
   * Process a single bundle + mode using the provided LayoutBuilderEntityViewDisplay as the template.
   * Returns an array with [total_nodes_found, updated_nodes_count].
   */
  private function processBundleMode(string $bundle, string $mode, LayoutBuilderEntityViewDisplay $template, bool $dryRun = FALSE): array {
    // Extract ws_header and ws_footer sections from the template layout.
    $default_sections = $template->getSections();
    [$default_header, $default_footer] = $this->getHeaderFooter($default_sections);

    if (!$default_header || !$default_footer) {
      $this->io()->writeln('ws_header or ws_footer section not found in the layout for ' . $bundle . '.' . $mode . '. Skipping.');
      return [0, 0];
    }

    // Query all nodes of this bundle.
    $nids = \Drupal::entityQuery('node')
      ->condition('type', $bundle)
      ->accessCheck(FALSE)
      ->execute();

    if (empty($nids)) {
      $this->io()->writeln('No ' . $bundle . ' nodes found for mode ' . $mode . '.');
      return [0, 0];
    }

    // Initialize counters.
    $total_nodes = 0;
    $updated_nodes = 0;

    // Load and update each node.
    $nodes = $this->entityTypeManager
      ->getStorage('node')
      ->loadMultiple($nids);
    foreach ($nodes as $node) {
      $total_nodes++;
      $layout = $node->get('layout_builder__layout');
      if ($layout) {
        $sections = $layout->getSections();

        // Determine whether applying the default header/footer would actually change the node.
        $would_change = FALSE;
        foreach ($sections as $s_index => $section) {
          /** @var \Drupal\layout_builder\Section $section */
          $section_id = $section->getLayoutId();
          if ($section_id === 'ws_header') {
            if ($section != $default_header) {
              $would_change = TRUE;
              // Replace only in the copy.
              $sections[$s_index] = $default_header;
            }
          }
          elseif ($section_id === 'ws_footer') {
            if ($section != $default_footer) {
              $would_change = TRUE;
              $sections[$s_index] = $default_footer;
            }
          }
        }

        if ($would_change) {
          // If not a dry run, persist changes. Otherwise, just report that it would change.
          if (!$dryRun) {
            $node->get('layout_builder__layout')->setValue($sections);
            $node->save();
          }
          $updated_nodes++;
        }
      }
    }

    return [$total_nodes, $updated_nodes];
  }

  /**
   * @param array $sections
   *
   * @return \Drupal\layout_builder\Section[]
   */
  private function getHeaderFooter(array $sections): array {
    $header = $footer = [];
    foreach ($sections as $section) {
      /** @var \Drupal\layout_builder\Section $section */
      $id = $section->getLayoutId();
      if ($id == 'ws_header') {
        $header = $section;
      }
      elseif ($id == 'ws_footer') {
        $footer = $section;
      }
    }
    return [$header, $footer];
  }

}
