<?php

/**
 * @file
 * Builds placeholder replacement tokens for Linked Content Entity.
 */

use Drupal\Core\Render\BubbleableMetadata;

/**
 * Implements hook_token_info().
 */
function lce_token_info() {
  $info = [];

  $info['types']['lce'] = [
    'name' => t('Linked Content Entity'),
    'description' => t('Set and retrieve linked content entity.'),
  ];

  $info['tokens']['lce']['label'] = [
    'name' => t('Label'),
    'description' => t('Gets the label of the linked content entity.'),
    'dynamic' => TRUE,
  ];

  $info['tokens']['lce']['path'] = [
    'name' => t('Path'),
    'description' => t('Gets the path to the linked content entity.'),
    'dynamic' => TRUE,
  ];

  $info['tokens']['lce']['url'] = [
    'name' => t('Url'),
    'description' => t('Gets the url to the linked content entity.'),
    'dynamic' => TRUE,
  ];

  $info['tokens']['lce']['id'] = [
    'name' => t('Id'),
    'description' => t('Gets the id of the linked content entity.'),
    'dynamic' => TRUE,
  ];

  return $info;
}

/**
 * Implements hook_tokens().
 */
function lce_tokens($type, $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata) {
  $language_manager = \Drupal::languageManager();
  $replacements = [];
  $url_options = [];

  if (isset($options['langcode'])) {
    $url_options['language'] = $language_manager->getLanguage($options['langcode']);
  }

  if ($type === 'lce') {
    foreach ($tokens as $name => $original) {
      [$name, $identifier] = explode(':', $name, 2) + [NULL, NULL];

      // If the identifier is not available, continue.
      if ($identifier === NULL) {
        $replacements[$original] = '';
        continue;
      }

      /** @var \Drupal\lce\LinkedContentEntityStorage $storage */
      $storage = \Drupal::entityTypeManager()->getStorage('linked_content_entity');

      /** @var \Drupal\lce\Entity\LinkedContentEntity $entity */
      $entity = $storage->load($identifier);

      // If the linked content entity is not found, continue.
      if (!$entity) {
        $replacements[$original] = '';
        continue;
      }

      // Add caching for the config entity.
      $bubbleable_metadata->addCacheableDependency($entity);

      /** @var \Drupal\Core\Entity\EntityInterface $content_entity */
      $content_entity = $storage->getContentEntityByEntity($entity);

      // If the content entity is not found, continue.
      if (!$content_entity) {
        $replacements[$original] = '';
        continue;
      }

      // Add caching for the linked content entity.
      $bubbleable_metadata->addCacheableDependency($content_entity);

      switch ($name) {
        case 'label':
          $replacements[$original] = $content_entity->label();
          break;

        case 'path':
          $alias_manager = \Drupal::service('path_alias.manager');
          $url = $content_entity->toUrl('canonical', $url_options);
          $path = '/' . $url->getInternalPath();
          $value = $alias_manager->getAliasByPath($path, $url->getOption('language')->getId());
          $replacements[$original] = $value;
          break;

        case 'url':
          $url = $content_entity->toUrl('canonical', $url_options)->toString(TRUE);
          $bubbleable_metadata->addCacheableDependency($url);
          $replacements[$original] = $url->getGeneratedUrl();
          break;

        case 'id':
          $replacements[$original] = $content_entity->id();
          break;
      }
    }
  }

  return $replacements;
}
