<?php

namespace Drupal\lce\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\lce\Entity\LinkedContentEntity;

/**
 * Form handler for the Linked Content Entity add and edit forms.
 */
class LinkedContentEntityForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $this->entity->label(),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $this->entity->id(),
      '#machine_name' => [
        'exists' => [LinkedContentEntity::class, 'load'],
        'source' => ['label'],
      ],
      '#disabled' => !$this->entity->isNew(),
    ];

    $form['content_entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Content Entity Type'),
      '#required' => TRUE,
      '#options' => $this->getEntityTypeOptions(),
      '#default_value' => $this->entity->getContentEntityType(),
      '#ajax' => [
        'callback' => [$this, 'updateType'],
        'event' => 'change',
        'effect' => 'fade',
        'wrapper' => 'content-entity-container',
      ],
    ];

    $form['content_entity_container'] = [
      '#type' => 'container',
      '#prefix' => '<div id="content-entity-container">',
      '#suffix' => '</div>',
    ];

    if ($type = $this->entity->getContentEntityType()) {
      $content_entity = $this->entityTypeManager
        ->getStorage('linked_content_entity')
        ->getContentEntityByEntity($this->entity);

      $form['content_entity_container']['content_entity_id'] = [
        '#type' => 'entity_autocomplete',
        '#title' => $this->t('Content Entity'),
        '#target_type' => $type,
        '#default_value' => $content_entity,
        '#required' => TRUE,
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $status = $this->entity->save();
    $t_args = ['%label' => $this->entity->label()];

    if ($status == SAVED_UPDATED) {
      $this->messenger()->addMessage($this->t('Linked Content Entity %label has been updated.', $t_args));
    }
    else {
      $this->messenger()->addMessage($this->t('Linked Content Entity %label has been added.', $t_args));
    }

    $form_state->setRedirect('entity.linked_content_entity.collection');

    return $status;
  }

  /**
   * Get list of entity types.
   *
   * @return array
   *   List of entity types.
   */
  protected function getEntityTypeOptions() {
    $options = [];
    $types = $this->entityTypeManager
      ->getDefinitions();

    foreach ($types as $type) {
      if ($type->hasViewBuilderClass() && $type->hasKey('uuid')) {
        $options[$type->get('id')] = $type->get('label');
      }
    }

    return $options;
  }

  /**
   * Handles the content entity id select element.
   *
   * @param array $form
   *   The form.
   *
   * @return array
   *   The container for the content entity id select element.
   */
  public function updateType(array &$form) {
    return $form['content_entity_container'];
  }

}
