<?php

namespace Drupal\lce;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Config\Entity\ConfigEntityStorage;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the linked content entity storage.
 */
class LinkedContentEntityStorage extends ConfigEntityStorage {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    $instance = parent::createInstance($container, $entity_type);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityRepository = $container->get('entity.repository');
    return $instance;
  }

  /**
   * Get the associated content entity.
   *
   * @param string $identifier
   *   The identifier to match.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   An entity object. NULL if no matching entity is found.
   */
  public function getContentEntity($identifier) {
    /** @var \Drupal\lce\LinkedContentEntityInterface $entity */
    $entity = $this->load($identifier);

    if (!$entity) {
      return NULL;
    }

    return $this->getContentEntityByEntity($entity);
  }

  /**
   * Get the associated content entity by the config entity.
   *
   * @param \Drupal\lce\LinkedContentEntityInterface $entity
   *   The config entity.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   An entity object. NULL if no matching entity is found.
   */
  public function getContentEntityByEntity(LinkedContentEntityInterface $entity): mixed {
    $type = $entity->getContentEntityType();
    $id = $entity->getContentEntityId();

    if (!$type || !$id) {
      return NULL;
    }

    return $this->getContentEntityByProperties($type, $id);
  }

  /**
   * Get the associated content entity by type and id.
   *
   * @param string $type
   *   The entity type.
   * @param mixed $id
   *   The entity id.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   An content object. NULL if no matching entity is found.
   */
  public function getContentEntityByProperties($type, $id): mixed {
    $content_entity = $this->entityTypeManager
      ->getStorage($type)
      ->load($id);

    return $this->entityRepository->getTranslationFromContext($content_entity);
  }

  /**
   * {@inheritdoc}
   */
  protected function doPostSave(EntityInterface $entity, $update) {
    if ($update) {
      foreach ([$entity, $entity->original] as $item) {
        $content_entity = $this->getContentEntityByEntity($item);

        if ($content_entity instanceof EntityInterface) {
          Cache::invalidateTags($content_entity->getCacheTagsToInvalidate());
        }
      }
    }

    parent::doPostSave($entity, $update);
  }

}
