<?php

namespace Drupal\lce\Twig;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

/**
 * Twig extension with some useful functions and filters.
 */
class TwigExtension extends AbstractExtension {

  /**
   * The linked content entity storage object.
   *
   * @var \Drupal\lce\LinkedContentEntityStorage
   */
  protected $storage;

  /**
   * Class constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entityTypeManager.
   * @param \Drupal\Core\Routing\UrlGeneratorInterface $urlGenerator
   *   The URL generator.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    protected UrlGeneratorInterface $urlGenerator,
    protected LanguageManagerInterface $languageManager,
  ) {
    $this->storage = $entity_type_manager->getStorage('linked_content_entity');
  }

  /**
   * {@inheritdoc}
   */
  public function getFunctions() {
    return [
      new TwigFunction('lce', [$this, 'getEntity']),
      new TwigFunction('lce_label', [$this, 'getLabel']),
      new TwigFunction('lce_url', [$this, 'getUrl']),
      new TwigFunction('lce_content_entity', [$this, 'getContentEntity']),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return 'lce';
  }

  /**
   * Get the config entity.
   *
   * @param string $identifier
   *   The identifier to get the config entity.
   *
   * @return \Drupal\lce\LinkedContentEntityInterface|null
   *   A linked content entity object. NULL if no matching entity is found.
   */
  public function getEntity($identifier) {
    return $this->storage->load($identifier);
  }

  /**
   * Get the label to the linked content entity.
   *
   * @param string $identifier
   *   The identifier of the linked content entity to retrieve.
   *
   * @return array|null
   *   The render array for content label. Null if no matching content is found.
   */
  public function getLabel($identifier) {
    /** @var \Drupal\lce\LinkedContentEntityInterface|null $entity */
    $entity = $this->storage->load($identifier);

    if (!$entity) {
      return NULL;
    }

    $build = [];

    // Add caching for the config entity.
    CacheableMetadata::createFromObject($entity)
      ->applyTo($build);

    $content_entity = $this->storage->getContentEntityByEntity($entity);

    if ($content_entity) {
      $build['#markup'] = $content_entity->label();

      // Add cacheable meta data to the build array.
      CacheableMetadata::createFromRenderArray($build)
        ->addCacheableDependency($content_entity)
        ->applyTo($build);
    }

    return $build;
  }

  /**
   * Get an url to the linked content entity.
   *
   * @param string $identifier
   *   The identifier of the linked content entity to retrieve.
   * @param string $rel
   *   The link relationship type, for example: canonical or edit-form.
   * @param array $options
   *   See \Drupal\Core\Routing\UrlGeneratorInterface::generateFromRoute() for
   *   the available options.
   *
   * @return array|null
   *   The render array for entity url. Null if no matching entity is found.
   */
  public function getUrl($identifier, string $rel = 'canonical', array $options = []) {
    /** @var \Drupal\lce\LinkedContentEntityInterface|null $entity */
    $entity = $this->storage->load($identifier);

    if (!$entity) {
      return NULL;
    }

    $build = [];

    // Add caching for the config entity.
    CacheableMetadata::createFromObject($entity)
      ->applyTo($build);

    $content_entity = $this->storage->getContentEntityByEntity($entity);

    if ($content_entity) {
      if (isset($options['langcode'])) {
        if ($language = $this->languageManager->getLanguage($options['langcode'])) {
          $options['language'] = $language;
        }
      }

      $url = $content_entity->toUrl($rel, $options)->toString(TRUE);
      $build['#markup'] = $url->getGeneratedUrl();

      // Add cacheable meta data to the build array.
      CacheableMetadata::createFromRenderArray($build)
        ->addCacheableDependency($url)
        ->addCacheableDependency($content_entity)
        ->applyTo($build);
    }

    return $build;
  }

  /**
   * Get the entity of the linked content entity.
   *
   * @param string $identifier
   *   The identifier of the linked content entity to retrieve.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   An entity object. NULL if no matching entity is found.
   */
  public function getContentEntity($identifier) {
    return $this->storage->getContentEntity($identifier);
  }

}
