/**
 * @file
 * LC Lightbox integration for Drupal.
 */

(function ($, Drupal, drupalSettings, once) {
  'use strict';

  /**
   * Initialize LC Lightbox galleries and handle auto-start behavior.
   */
  Drupal.behaviors.lclLightbox = {
    attach: function (context, settings) {

      // Check if LC Lightbox library is available
      if (typeof window.lc_lightbox === 'undefined') {
        console.error('LC Lightbox library is not loaded. The lc_lightbox function is not available.');
        return;
      }

      // Process each gallery defined in settings.
      if (settings.lcl && settings.lcl.galleries) {
        Object.keys(settings.lcl.galleries).forEach(function (galleryId) {
          var galleryConfig = settings.lcl.galleries[galleryId];
          var profileId = galleryConfig.profile;
          var autoStart = galleryConfig.auto_start;

          // Get profile configuration.
          var profileSettings = settings.lcl.profiles && settings.lcl.profiles[profileId]
            ? settings.lcl.profiles[profileId]
            : {};

          // Find gallery elements.
          var gallerySelector = '[data-lcl-lightbox="' + galleryId + '"]';
          var $galleryElements = $(gallerySelector, context);

          if ($galleryElements.length === 0) {
            return;
          }

          // Initialize the entire gallery at once (not individual elements).
          if (!$galleryElements.first().attr('data-lcl-gallery-initialized')) {
            Drupal.lclLightbox.initializeGallery(gallerySelector, profileSettings, autoStart, galleryId);
            $galleryElements.attr('data-lcl-gallery-initialized', 'true');
          }
        });
      }

      // Handle individual lightbox elements (fallback for elements without gallery config).
      once('lcl-lightbox-individual', '[data-lcl-lightbox]:not([data-lcl-processed])', context).forEach(function (element) {
        var $element = $(element);
        var profileId = $element.data('lcl-profile') || 'default';
        var profileSettings = settings.lcl && settings.lcl.profiles && settings.lcl.profiles[profileId]
          ? settings.lcl.profiles[profileId]
          : Drupal.lclLightbox.getDefaultSettings();

        // Mark as processed to avoid double initialization.
        $element.attr('data-lcl-processed', 'true');

        // Initialize individual element.
        Drupal.lclLightbox.initializeElement($element, profileSettings);
      });
    }
  };

  /**
   * LC Lightbox utility functions.
   */
  Drupal.lclLightbox = {

    /**
     * Initialize a gallery of LC Lightbox elements.
     *
     * @param {string} selector
     *   CSS selector for gallery elements.
     * @param {Object} profileSettings
     *   Profile configuration object.
     * @param {boolean} autoStart
     *   Whether to auto-start the lightbox.
     * @param {string} galleryId
     *   Unique gallery identifier.
     */
    initializeGallery: function (selector, profileSettings, autoStart, galleryId) {
      var lightboxSettings = $.extend({}, this.getDefaultSettings(), profileSettings);
      var $elements = $(selector);

      // Ensure gallery mode is enabled and set the correct gallery hook.
      lightboxSettings.gallery = true;
      lightboxSettings.gallery_hook = 'data-lcl-lightbox';

      // Mark elements as processed.
      $elements.attr('data-lcl-processed', 'true');

      // Initialize LC Lightbox using the global lc_lightbox function with selector string.
      window.lc_lightbox(selector, lightboxSettings);

      // Handle auto-start behavior.
      if (autoStart && $elements.length > 0) {
        // Auto-start by triggering click on the first element after a short delay.
        setTimeout(function () {
          $elements.first().trigger('click');
        }, 100);
      }
    },

    /**
     * Initialize a single LC Lightbox element.
     *
     * @param {jQuery} $element
     *   jQuery object containing the element.
     * @param {Object} profileSettings
     *   Profile configuration object.
     */
    initializeElement: function ($element, profileSettings) {
      var lightboxSettings = $.extend({}, this.getDefaultSettings(), profileSettings);

      // Initialize LC Lightbox using the global lc_lightbox function.
      window.lc_lightbox($element, lightboxSettings);
    },

    /**
     * Get default LC Lightbox settings.
     *
     * @return {Object}
     *   Default settings object.
     */
    getDefaultSettings: function () {
      return {
        gallery: true,
        gallery_hook: 'data-lcl-lightbox',
        live_elements: true,
        preload_all: false,
        deeplink: false,
        modal: false,
        skin: 'light',
        data_position: 'over',
        cmd_position: 'inner',
        ins_close_pos: 'normal',
        nav_btn_pos: 'normal',
        max_width: '93%',
        max_height: '93%',
        border_w: 0,
        border_col: '#ddd',
        padding: 0,
        radius: 0,
        ol_opacity: 0.7,
        ol_color: '#111',
        shadow: true,
        remove_scrollbar: true,
        open_close_time: 500,
        fading_time: 150,
        animation_time: 300,
        slideshow: true,
        autoplay: false,
        slideshow_time: 6000,
        counter: false,
        progressbar: true,
        carousel: true,
        show_title: true,
        show_descr: true,
        show_author: true,
        txt_toggle_cmd: true,
        thumbs_nav: true,
        tn_icons: true,
        thumbs_w: 110,
        thumbs_h: 110,
        fullscreen: true,
        fs_img_behavior: 'fit',
        browser_fs_mode: true,
        socials: true,
        download: false,
        touchswipe: true,
        mousewheel: true,
        autoplay_videos: false,
        rclick_prevent: false,
        img_zoom: false,
        txt_hidden: 500,
        tn_hidden: 500,
        fs_only: 500
      };
    },

    /**
     * Programmatically initialize LC Lightbox on elements.
     *
     * @param {jQuery} $elements
     *   jQuery object containing elements to initialize.
     * @param {Object} settings
     *   LC Lightbox settings object.
     */
    init: function ($elements, settings) {
      settings = $.extend({}, this.getDefaultSettings(), settings || {});
      window.lc_lightbox($elements, settings);
    }
  };

})(jQuery, Drupal, drupalSettings, once);
