<?php

declare(strict_types=1);

namespace Drupal\lcl\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\lcl\Entity\LcLightboxProfile;

/**
 * LC Lightbox Profiles form.
 */
final class LcLightboxProfileForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {

    $form = parent::form($form, $form_state);

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $this->entity->label(),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $this->entity->id(),
      '#machine_name' => [
        'exists' => [LcLightboxProfile::class, 'load'],
      ],
      '#disabled' => !$this->entity->isNew(),
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $this->entity->status(),
    ];

    // Core Behavior Settings
    $form['core_behavior'] = [
      '#type' => 'details',
      '#title' => $this->t('Core Behavior'),
      '#open' => TRUE,
    ];

    $form['core_behavior']['gallery'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Gallery Mode'),
      '#description' => $this->t('Display multiple elements as a gallery instead of individually.'),
      '#default_value' => $this->entity->get('gallery') ?? TRUE,
    ];

    $form['core_behavior']['gallery_hook'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Gallery Hook Attribute'),
      '#description' => $this->t('Attribute used to group elements. Use "false" to create a gallery with all fetched elements.'),
      '#default_value' => $this->entity->get('gallery_hook') ?? 'rel',
      '#size' => 20,
    ];

    $form['core_behavior']['live_elements'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Live Elements Tracking'),
      '#description' => $this->t('Track DOM changes to dynamically alter fetched elements.'),
      '#default_value' => $this->entity->get('live_elements') ?? TRUE,
    ];

    $form['core_behavior']['preload_all'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Preload All Images'),
      '#description' => $this->t('Preload all images on document ready.'),
      '#default_value' => $this->entity->get('preload_all') ?? FALSE,
    ];

    $form['core_behavior']['deeplink'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Deep Linking'),
      '#description' => $this->t('Create browser history and unique URLs for every element.'),
      '#default_value' => $this->entity->get('deeplink') ?? FALSE,
    ];

    $form['core_behavior']['modal'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Modal Mode'),
      '#description' => $this->t('Enable modal mode (no lightbox closing on overlay click).'),
      '#default_value' => $this->entity->get('modal') ?? FALSE,
    ];

    // Visual & Layout Settings
    $form['visual_layout'] = [
      '#type' => 'details',
      '#title' => $this->t('Visual & Layout'),
      '#open' => FALSE,
    ];

    $form['visual_layout']['skin'] = [
      '#type' => 'select',
      '#title' => $this->t('Skin'),
      '#description' => $this->t('Visual theme for the lightbox.'),
      '#options' => [
        'light' => $this->t('Light'),
        'dark' => $this->t('Dark'),
        'minimal' => $this->t('Minimal'),
      ],
      '#default_value' => $this->entity->get('skin') ?? 'light',
    ];

    $form['visual_layout']['data_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Data Position'),
      '#description' => $this->t('Where element data (title, description) will be shown.'),
      '#options' => [
        'over' => $this->t('Over element'),
        'under' => $this->t('Under element'),
        'rside' => $this->t('Right side'),
        'lside' => $this->t('Left side'),
        'hidden' => $this->t('Hidden'),
      ],
      '#default_value' => $this->entity->get('data_position') ?? 'over',
    ];

    $form['visual_layout']['cmd_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Commands Position'),
      '#description' => $this->t('Where navigation commands will be displayed.'),
      '#options' => [
        'inner' => $this->t('Inside lightbox'),
        'outer' => $this->t('Outside lightbox'),
      ],
      '#default_value' => $this->entity->get('cmd_position') ?? 'inner',
    ];

    $form['visual_layout']['ins_close_pos'] = [
      '#type' => 'select',
      '#title' => $this->t('Close Button Position'),
      '#description' => $this->t('Position of the close button.'),
      '#options' => [
        'normal' => $this->t('Normal'),
        'corner' => $this->t('Corner (only for inner commands)'),
      ],
      '#default_value' => $this->entity->get('ins_close_pos') ?? 'normal',
    ];

    $form['visual_layout']['nav_btn_pos'] = [
      '#type' => 'select',
      '#title' => $this->t('Navigation Buttons Position'),
      '#description' => $this->t('Position of the navigation arrows.'),
      '#options' => [
        'normal' => $this->t('Normal'),
        'middle' => $this->t('Middle'),
      ],
      '#default_value' => $this->entity->get('nav_btn_pos') ?? 'normal',
    ];

    $form['visual_layout']['opening_effect'] = [
      '#type' => 'select',
      '#title' => $this->t('Opening/Closing Effect'),
      '#description' => $this->t('Animation effect for opening and closing the lightbox.'),
      '#options' => [
        'fade' => $this->t('Fade'),
        'zoom-in' => $this->t('Zoom In'),
        'bottom-to-top' => $this->t('Bottom to Top'),
        'bottom-to-top-v2' => $this->t('Bottom to Top (v2)'),
        'right-to-left' => $this->t('Right to Left'),
        'horizontal-flip' => $this->t('Horizontal Flip'),
        'vertical-flip' => $this->t('Vertical Flip'),
        'none' => $this->t('None'),
      ],
      '#default_value' => $this->entity->get('opening_effect') ?? 'fade',
    ];

    // Sizing & Dimensions
    $form['sizing'] = [
      '#type' => 'details',
      '#title' => $this->t('Sizing & Dimensions'),
      '#open' => FALSE,
    ];

    $form['sizing']['max_width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Maximum Width'),
      '#description' => $this->t('Maximum width (use % for responsive or px for fixed).'),
      '#default_value' => $this->entity->get('max_width') ?? '93%',
      '#size' => 10,
    ];

    $form['sizing']['max_height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Maximum Height'),
      '#description' => $this->t('Maximum height (use % for responsive or px for fixed).'),
      '#default_value' => $this->entity->get('max_height') ?? '93%',
      '#size' => 10,
    ];

    $form['sizing']['border_w'] = [
      '#type' => 'number',
      '#title' => $this->t('Border Width'),
      '#description' => $this->t('Border width in pixels.'),
      '#default_value' => $this->entity->get('border_w') ?? 0,
      '#min' => 0,
      '#max' => 50,
    ];

    $form['sizing']['padding'] = [
      '#type' => 'number',
      '#title' => $this->t('Content Padding'),
      '#description' => $this->t('Content padding in pixels.'),
      '#default_value' => $this->entity->get('padding') ?? 0,
      '#min' => 0,
      '#max' => 100,
    ];

    $form['sizing']['radius'] = [
      '#type' => 'number',
      '#title' => $this->t('Border Radius'),
      '#description' => $this->t('Border radius in pixels.'),
      '#default_value' => $this->entity->get('radius') ?? 0,
      '#min' => 0,
      '#max' => 50,
    ];

    // Animation & Timing
    $form['timing'] = [
      '#type' => 'details',
      '#title' => $this->t('Animation & Timing'),
      '#open' => FALSE,
    ];

    $form['timing']['open_close_time'] = [
      '#type' => 'number',
      '#title' => $this->t('Open/Close Time'),
      '#description' => $this->t('Animation duration for opening and closing (milliseconds).'),
      '#default_value' => $this->entity->get('open_close_time') ?? 500,
      '#min' => 0,
      '#max' => 5000,
      '#step' => 50,
    ];

    $form['timing']['fading_time'] = [
      '#type' => 'number',
      '#title' => $this->t('Fading Time'),
      '#description' => $this->t('Elements fading animation duration (milliseconds).'),
      '#default_value' => $this->entity->get('fading_time') ?? 150,
      '#min' => 0,
      '#max' => 2000,
      '#step' => 50,
    ];

    $form['timing']['animation_time'] = [
      '#type' => 'number',
      '#title' => $this->t('Animation Time'),
      '#description' => $this->t('Sizing animation duration (milliseconds).'),
      '#default_value' => $this->entity->get('animation_time') ?? 300,
      '#min' => 0,
      '#max' => 2000,
      '#step' => 50,
    ];

    // Slideshow Settings
    $form['slideshow'] = [
      '#type' => 'details',
      '#title' => $this->t('Slideshow'),
      '#open' => FALSE,
    ];

    $form['slideshow']['slideshow_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Slideshow'),
      '#description' => $this->t('Enable slideshow system.'),
      '#default_value' => $this->entity->get('slideshow') ?? TRUE,
    ];

    $form['slideshow']['autoplay'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Autoplay'),
      '#description' => $this->t('Automatically start slideshow.'),
      '#default_value' => $this->entity->get('autoplay') ?? FALSE,
      '#states' => [
        'visible' => [
          ':input[name="slideshow_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['slideshow']['slideshow_time'] = [
      '#type' => 'number',
      '#title' => $this->t('Slideshow Interval'),
      '#description' => $this->t('Time between slides (milliseconds).'),
      '#default_value' => $this->entity->get('slideshow_time') ?? 6000,
      '#min' => 1000,
      '#max' => 30000,
      '#step' => 500,
      '#states' => [
        'visible' => [
          ':input[name="slideshow_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['slideshow']['progressbar'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Progress Bar'),
      '#description' => $this->t('Display progress bar during slideshow.'),
      '#default_value' => $this->entity->get('progressbar') ?? TRUE,
      '#states' => [
        'visible' => [
          ':input[name="slideshow_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['slideshow']['counter'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Elements Counter'),
      '#description' => $this->t('Display current element number and total count.'),
      '#default_value' => $this->entity->get('counter') ?? FALSE,
    ];

    $form['slideshow']['carousel'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Carousel Mode'),
      '#description' => $this->t('Create non-stop pagination cycling through elements.'),
      '#default_value' => $this->entity->get('carousel') ?? TRUE,
    ];

    // Thumbnails Navigation
    $form['thumbnails'] = [
      '#type' => 'details',
      '#title' => $this->t('Thumbnails'),
      '#open' => FALSE,
    ];

    $form['thumbnails']['thumbs_nav'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Thumbnails Navigation'),
      '#description' => $this->t('Show thumbnails navigation bar.'),
      '#default_value' => $this->entity->get('thumbs_nav') ?? TRUE,
    ];

    $form['thumbnails']['thumbs_w'] = [
      '#type' => 'number',
      '#title' => $this->t('Thumbnail Width'),
      '#description' => $this->t('Thumbnail width in pixels.'),
      '#default_value' => $this->entity->get('thumbs_w') ?? 110,
      '#min' => 50,
      '#max' => 300,
      '#states' => [
        'visible' => [
          ':input[name="thumbs_nav"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['thumbnails']['thumbs_h'] = [
      '#type' => 'number',
      '#title' => $this->t('Thumbnail Height'),
      '#description' => $this->t('Thumbnail height in pixels.'),
      '#default_value' => $this->entity->get('thumbs_h') ?? 110,
      '#min' => 50,
      '#max' => 300,
      '#states' => [
        'visible' => [
          ':input[name="thumbs_nav"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['thumbnails']['tn_icons'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Type Icons'),
      '#description' => $this->t('Show element type icons on thumbnails.'),
      '#default_value' => $this->entity->get('tn_icons') ?? TRUE,
      '#states' => [
        'visible' => [
          ':input[name="thumbs_nav"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Fullscreen Settings
    $form['fullscreen'] = [
      '#type' => 'details',
      '#title' => $this->t('Fullscreen'),
      '#open' => FALSE,
    ];

    $form['fullscreen']['fullscreen_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Fullscreen'),
      '#description' => $this->t('Allow fullscreen mode.'),
      '#default_value' => $this->entity->get('fullscreen') ?? TRUE,
    ];

    $form['fullscreen']['fs_img_behavior'] = [
      '#type' => 'select',
      '#title' => $this->t('Fullscreen Image Behavior'),
      '#description' => $this->t('How images behave in fullscreen mode.'),
      '#options' => [
        'fit' => $this->t('Fit - maintain aspect ratio'),
        'fill' => $this->t('Fill - stretch to fill screen'),
        'smart' => $this->t('Smart - best fit based on image'),
      ],
      '#default_value' => $this->entity->get('fs_img_behavior') ?? 'fit',
      '#states' => [
        'visible' => [
          ':input[name="fullscreen_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['fullscreen']['browser_fs_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Browser Fullscreen Mode'),
      '#description' => $this->t('Use browser native fullscreen mode.'),
      '#default_value' => $this->entity->get('browser_fs_mode') ?? TRUE,
      '#states' => [
        'visible' => [
          ':input[name="fullscreen_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Content Display
    $form['content'] = [
      '#type' => 'details',
      '#title' => $this->t('Content Display'),
      '#open' => FALSE,
    ];

    $form['content']['show_title'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Titles'),
      '#description' => $this->t('Display element titles.'),
      '#default_value' => $this->entity->get('show_title') ?? TRUE,
    ];

    $form['content']['show_descr'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Descriptions'),
      '#description' => $this->t('Display element descriptions.'),
      '#default_value' => $this->entity->get('show_descr') ?? TRUE,
    ];

    $form['content']['show_author'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Authors'),
      '#description' => $this->t('Display element authors.'),
      '#default_value' => $this->entity->get('show_author') ?? TRUE,
    ];

    $form['content']['txt_toggle_cmd'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Text Toggle Button'),
      '#description' => $this->t('Show button to toggle text visibility.'),
      '#default_value' => $this->entity->get('txt_toggle_cmd') ?? TRUE,
    ];

    // Overlay Settings
    $form['overlay'] = [
      '#type' => 'details',
      '#title' => $this->t('Overlay'),
      '#open' => FALSE,
    ];

    $form['overlay']['ol_opacity'] = [
      '#type' => 'number',
      '#title' => $this->t('Overlay Opacity'),
      '#description' => $this->t('Overlay opacity (0.0 to 1.0).'),
      '#default_value' => $this->entity->get('ol_opacity') ?? 0.7,
      '#min' => 0,
      '#max' => 1,
      '#step' => 0.1,
    ];

    $form['overlay']['ol_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Overlay Color'),
      '#description' => $this->t('Background overlay color.'),
      '#default_value' => $this->entity->get('ol_color') ?? '#111111',
    ];

    $form['overlay']['shadow'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Shadow'),
      '#description' => $this->t('Apply shadow around lightbox window.'),
      '#default_value' => $this->entity->get('shadow') ?? TRUE,
    ];

    $form['overlay']['remove_scrollbar'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Remove Scrollbar'),
      '#description' => $this->t('Hide page scrollbar when lightbox is open.'),
      '#default_value' => $this->entity->get('remove_scrollbar') ?? TRUE,
    ];

    // Social & Interaction
    $form['social'] = [
      '#type' => 'details',
      '#title' => $this->t('Social & Interaction'),
      '#open' => FALSE,
    ];

    $form['social']['socials'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Social Sharing'),
      '#description' => $this->t('Show social share buttons.'),
      '#default_value' => $this->entity->get('socials') ?? TRUE,
    ];

    $form['social']['download'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Download Button'),
      '#description' => $this->t('Show download button for elements.'),
      '#default_value' => $this->entity->get('download') ?? FALSE,
    ];

    $form['social']['touchswipe'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Touch/Swipe Navigation'),
      '#description' => $this->t('Enable touch gestures and swipe navigation.'),
      '#default_value' => $this->entity->get('touchswipe') ?? TRUE,
    ];

    $form['social']['mousewheel'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Mousewheel Navigation'),
      '#description' => $this->t('Navigate elements using mousewheel.'),
      '#default_value' => $this->entity->get('mousewheel') ?? TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    // Map form field names to entity properties where they differ
    $field_mapping = [
      'slideshow_enabled' => 'slideshow',
      'fullscreen_enabled' => 'fullscreen',
    ];

    // Set all form values to the entity
    $values = $form_state->getValues();
    foreach ($values as $key => $value) {
      // Skip form API specific keys
      if (in_array($key, ['form_build_id', 'form_token', 'form_id', 'op', 'actions'])) {
        continue;
      }

      // Use mapped field name if exists
      $entity_key = $field_mapping[$key] ?? $key;

      // Handle special cases
      if ($key === 'ol_color' && !empty($value)) {
        // Ensure color values are properly formatted
        $value = ltrim($value, '#');
        $value = '#' . $value;
      }

      $this->entity->set($entity_key, $value);
    }

    $result = parent::save($form, $form_state);
    $message_args = ['%label' => $this->entity->label()];
    $this->messenger()->addStatus(
      match($result) {
        \SAVED_NEW => $this->t('Created new LC Lightbox profile %label.', $message_args),
        \SAVED_UPDATED => $this->t('Updated LC Lightbox profile %label.', $message_args),
      }
    );
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // Validate max_width and max_height formats
    $width = $form_state->getValue('max_width');
    $height = $form_state->getValue('max_height');

    if (!empty($width) && !preg_match('/^\d+(%|px)$/', $width)) {
      $form_state->setErrorByName('max_width', $this->t('Maximum width must be in format like "90%" or "800px".'));
    }

    if (!empty($height) && !preg_match('/^\d+(%|px)$/', $height)) {
      $form_state->setErrorByName('max_height', $this->t('Maximum height must be in format like "90%" or "600px".'));
    }

    // Validate opacity range
    $opacity = $form_state->getValue('ol_opacity');
    if ($opacity < 0 || $opacity > 1) {
      $form_state->setErrorByName('ol_opacity', $this->t('Overlay opacity must be between 0.0 and 1.0.'));
    }

    // Validate timing values
    $timing_fields = ['open_close_time', 'fading_time', 'animation_time', 'slideshow_time'];
    foreach ($timing_fields as $field) {
      $value = $form_state->getValue($field);
      if (!empty($value) && $value < 0) {
        $form_state->setErrorByName($field, $this->t('@field must be a positive number.', ['@field' => $form[$field]['#title']]));
      }
    }
  }

}
