<?php

declare(strict_types=1);

namespace Drupal\lcl\Composer;

use Composer\Script\Event;
use Composer\Util\Filesystem;

/**
 * Composer script to automatically install LC Lightbox library.
 */
class LibraryInstaller {

  /**
   * Install the LC Lightbox library.
   *
   * @param \Composer\Script\Event $event
   *   The Composer event.
   */
  public static function install(Event $event): void {
    $io = $event->getIO();
    $composer = $event->getComposer();
    $extra = $composer->getPackage()->getExtra();

    // Check if library installer configuration exists.
    if (!isset($extra['library-installer']['lc-lightbox-lite'])) {
      return;
    }

    $config = $extra['library-installer']['lc-lightbox-lite'];
    $url = $config['url'];
    $path = $config['path'];

    // Skip if library already exists.
    if (is_dir($path) && file_exists($path . '/js/lc_lightbox.lite.min.js')) {
      $io->write('<info>LC Lightbox library already installed.</info>');
      return;
    }

    $io->write('<info>Installing LC Lightbox library...</info>');

    try {
      // Create libraries directory if it doesn't exist.
      $filesystem = new Filesystem();
      $filesystem->ensureDirectoryExists(dirname($path));

      // Download the library.
      $tempFile = tempnam(sys_get_temp_dir(), 'lc-lightbox');
      if (!$tempFile || !copy($url, $tempFile)) {
        throw new \Exception('Failed to download LC Lightbox library.');
      }

      // Extract the ZIP file.
      $zip = new \ZipArchive();
      if ($zip->open($tempFile) !== TRUE) {
        throw new \Exception('Failed to open downloaded ZIP file.');
      }

      // Create temporary extraction directory.
      $tempDir = sys_get_temp_dir() . '/lc-lightbox-' . uniqid();
      $filesystem->ensureDirectoryExists($tempDir);

      // Extract to temporary directory.
      $zip->extractTo($tempDir);
      $zip->close();

      // Find the extracted directory (should be LC-Lightbox-Lite-master).
      $extractedDir = null;
      $iterator = new \DirectoryIterator($tempDir);
      foreach ($iterator as $item) {
        if ($item->isDir() && !$item->isDot() && strpos($item->getFilename(), 'LC-Lightbox-Lite') === 0) {
          $extractedDir = $item->getPathname();
          break;
        }
      }

      if (!$extractedDir) {
        throw new \Exception('Could not find extracted LC Lightbox directory.');
      }

      // Remove existing directory if it exists.
      if (is_dir($path)) {
        $filesystem->removeDirectory($path);
      }

      // Move to final location.
      if (!rename($extractedDir, $path)) {
        throw new \Exception('Failed to move LC Lightbox library to final location.');
      }

      // Clean up temporary files.
      unlink($tempFile);
      $filesystem->removeDirectory($tempDir);

      // Verify installation.
      if (file_exists($path . '/js/lc_lightbox.lite.min.js')) {
        $io->write('<info>LC Lightbox library installed successfully!</info>');
      } else {
        throw new \Exception('LC Lightbox library installation verification failed.');
      }

    } catch (\Exception $e) {
      $io->writeError('<error>Failed to install LC Lightbox library: ' . $e->getMessage() . '</error>');
      $io->writeError('<error>Please install manually: https://github.com/LCweb-ita/LC-Lightbox-Lite</error>');
    }
  }

}
