<?php

declare(strict_types=1);

namespace Drupal\lcl\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\image\Plugin\Field\FieldFormatter\ImageFormatterBase;
use Drupal\image\Plugin\Field\FieldFormatter\ImageFormatter;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\image\Entity\ImageStyle;
use Drupal\Core\Asset\LibraryDiscoveryInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'LC Lightbox' formatter.
 *
 * @FieldFormatter(
 *   id = "lc_lightbox",
 *   module = "lcl",
 *   label = @Translation("LC Lightbox"),
 *   field_types = {
 *     "image"
 *   }
 * )
 */
final class LightboxFormatter extends ImageFormatter implements ContainerFactoryPluginInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The entity type manager service object.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Element attachment allowing library to be attached to pages.
   *
   * @var \Drupal\colorbox\ElementAttachmentInterface
   */
  protected $attachment;

  /**
   * Drupal\Core\Extension\ModuleHandlerInterface definition.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  private $moduleHandler;

  /**
   * Library discovery service.
   *
   * @var \Drupal\Core\Asset\LibraryDiscoveryInterface
   */
  private $libraryDiscovery;

  /**
   * Constructs an ImageFormatter object.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The image style storage.
   * @param \Drupal\colorbox\ElementAttachmentInterface $attachment
   *   Allow the library to be attached to the page.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   Module handler services.
   * @param \Drupal\Core\Asset\LibraryDiscoveryInterface $libraryDiscovery
   *   Library discovery service.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    AccountInterface $current_user,
    EntityTypeManagerInterface $entity_type_manager,
    ModuleHandlerInterface $moduleHandler,
    LibraryDiscoveryInterface $libraryDiscovery,
    FileUrlGeneratorInterface $file_url_generator,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings, $current_user, $entity_type_manager->getStorage('image_style'), $file_url_generator);
    $this->currentUser = $current_user;
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleHandler = $moduleHandler;
    $this->libraryDiscovery = $libraryDiscovery;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
      $container->get('library.discovery'),
      $container->get('file_url_generator')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    $setting = [
      'lcl_image_style' => NULL,
      'lcl_profile' => NULL,
      'lcl_auto_start' => 'click',
      'image_loading' => [
        'attribute' => 'lazy',
      ],
    ];
    return $setting + parent::defaultSettings();
  }
  
  /**
   * Returns LC Lightbox profiles.
   */
  private function getLclProfiles() {
    $profiles = [];
    $lcl_profiles = $this->entityTypeManager->getStorage('lc_lightbox_profile')->loadMultiple();

    foreach ($lcl_profiles as $id => $profile) {
      $profiles[$id] = $profile->label();
    }

    return $profiles;
  }
   
  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $elements = parent::settingsForm($form, $form_state);
    $image_styles = image_style_options(FALSE);
    $profiles = $this->getLclProfiles();
    $image_styles_link = Link::fromTextAndUrl(
      $this->t('Configure Image Styles'),
      Url::fromRoute('entity.image_style.collection')
    );
    $lcl_profiles_link = Link::fromTextAndUrl(
      $this->t('Configure LC Lightbox Profiles'),
      Url::fromRoute('entity.lc_lightbox_profile.collection')
    );
    $elements['lcl_profile'] = [
      '#title' => $this->t('LC Lightbox Profile'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('lcl_profile'),
      '#empty_option' => $this->t('- Select -'),
      '#options' => $profiles,
      '#required' => TRUE,
      '#description' => $lcl_profiles_link->toRenderable() + [
        '#access' => $this->currentUser->hasPermission('administer lc_lightbox_profile'),
      ],
    ];
    $elements['lcl_image_style'] = [
      '#title' => $this->t('Image style for LC Lightbox'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('lcl_image_style'),
      '#empty_option' => $this->t('None (original image)'),
      '#options' => $image_styles,
      '#description' => $image_styles_link->toRenderable() + [
        '#access' => $this->currentUser->hasPermission('administer image styles'),
      ],
    ];

    $elements['lcl_auto_start'] = [
      '#title' => $this->t('Lightbox Activation'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('lcl_auto_start'),
      '#options' => [
        'click' => $this->t('On image click (default)'),
        'auto' => $this->t('Automatically on page load'),
      ],
      '#description' => $this->t('Choose when the lightbox should be activated. "On image click" is the standard behavior, while "Automatically on page load" will open the first image immediately when the page loads.'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = parent::settingsSummary();
    $image_styles = image_style_options(FALSE);
    $profiles = $this->getLclProfiles();

    if ($this->getSetting('lcl_profile')) {
      $summary[] = $this->t('LC Lightbox Profile: @profile', ['@profile' => $profiles[$this->getSetting('lcl_profile')]]);
    }

    // Unset possible 'No defined styles' option.
    unset($image_styles['']);
    if (isset($image_styles[$this->getSetting('lcl_image_style')])) {
      $summary[] = $this->t('LCL Image style: @style', ['@style' => $image_styles[$this->getSetting('lcl_image_style')]]);
    }

    $auto_start_options = [
      'click' => $this->t('On image click'),
      'auto' => $this->t('Automatically on page load'),
    ];
    if ($this->getSetting('lcl_auto_start')) {
      $summary[] = $this->t('Activation: @mode', ['@mode' => $auto_start_options[$this->getSetting('lcl_auto_start')]]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $elements = parent::viewElements($items, $langcode);
    $files = $this->getEntitiesToView($items, $langcode);

    // Early opt-out if the field is empty.
    if (empty($files)) {
      return $elements;
    }

    $profile_id = $this->getSetting('lcl_profile');
    $auto_start = $this->getSetting('lcl_auto_start');
    $lcl_image_style = $this->getSetting('lcl_image_style');

    // Load the selected profile.
    $profile = NULL;
    if ($profile_id) {
      $profile = $this->entityTypeManager->getStorage('lc_lightbox_profile')->load($profile_id);
    }

    // If no profile is selected or found, return regular elements.
    if (!$profile) {
      return $elements;
    }

    // Generate unique gallery ID for this field instance.
    $gallery_id = 'lcl-gallery-' . $this->fieldDefinition->getName() . '-' . uniqid();

    // Prepare profile configuration for JavaScript.
    $profile_config = $this->getProfileConfiguration($profile);

    // Add auto-start behavior to profile config.
    $profile_config['auto_start'] = ($auto_start === 'auto');
    $profile_config['gallery_id'] = $gallery_id;

    // Process each image element.
    foreach ($elements as $delta => $element) {
      if (!isset($files[$delta])) {
        continue;
      }

      $file = $files[$delta];
      $image_uri = $file->getFileUri();

      // Generate the lightbox image URL.
      $lightbox_url = $image_uri;
      if ($lcl_image_style) {
        $image_style = ImageStyle::load($lcl_image_style);
        if ($image_style) {
          $lightbox_url = $image_style->buildUrl($image_uri);
        }
      } else {
        $lightbox_url = $this->fileUrlGenerator->generateAbsoluteString($image_uri);
      }

      // Prepare link attributes for LC Lightbox.
      $link_attributes = [
        'data-lcl-lightbox' => $gallery_id,
        'data-lcl-profile' => $profile_id,
        'href' => $lightbox_url,
        'class' => ['lcl-lightbox-trigger'],
      ];

      // Add title and description if available.
      if (!empty($element['#item']->title)) {
        $link_attributes['title'] = $element['#item']->title;
      }
      if (!empty($element['#item']->alt)) {
        $link_attributes['data-lcl-txt'] = $element['#item']->alt;
      }

      // Always wrap image in a link for LC Lightbox.
      $elements[$delta] = [
        '#type' => 'link',
        '#title' => $elements[$delta],
        '#url' => Url::fromUri($lightbox_url),
        '#attributes' => $link_attributes,
      ];
    }

    // Attach the LC Lightbox library and pass configuration to JavaScript.
    $elements['#attached']['library'][] = 'lcl/lc_lightbox_drupal';
    $elements['#attached']['drupalSettings']['lcl']['profiles'][$profile_id] = $profile_config;
    $elements['#attached']['drupalSettings']['lcl']['galleries'][$gallery_id] = [
      'profile' => $profile_id,
      'auto_start' => ($auto_start === 'auto'),
    ];

    return $elements;
  }

  /**
   * Extract profile configuration for JavaScript.
   *
   * @param \Drupal\lcl\Entity\LcLightboxProfile $profile
   *   The LC Lightbox profile entity.
   *
   * @return array
   *   Configuration array for LC Lightbox JavaScript.
   */
  private function getProfileConfiguration($profile): array {
    $config = [];

    // Core behavior options.
    $config['gallery'] = (bool) $profile->get('gallery');
    $config['gallery_hook'] = $profile->get('gallery_hook') ?: 'rel';
    $config['live_elements'] = (bool) $profile->get('live_elements');
    $config['preload_all'] = (bool) $profile->get('preload_all');
    $config['deeplink'] = (bool) $profile->get('deeplink');
    $config['modal'] = (bool) $profile->get('modal');

    // Visual and layout options.
    $config['skin'] = $profile->get('skin') ?: 'light';
    $config['data_position'] = $profile->get('data_position') ?: 'over';
    $config['cmd_position'] = $profile->get('cmd_position') ?: 'inner';
    $config['ins_close_pos'] = $profile->get('ins_close_pos') ?: 'normal';
    $config['nav_btn_pos'] = $profile->get('nav_btn_pos') ?: 'normal';

    // Sizing options.
    $config['max_width'] = $profile->get('max_width') ?: '93%';
    $config['max_height'] = $profile->get('max_height') ?: '93%';
    $config['border_w'] = (int) $profile->get('border_w');
    $config['border_col'] = $profile->get('border_col') ?: '#ddd';
    $config['padding'] = (int) $profile->get('padding');
    $config['radius'] = (int) $profile->get('radius');

    // Overlay options.
    $config['ol_opacity'] = (float) $profile->get('ol_opacity') ?: 0.7;
    $config['ol_color'] = $profile->get('ol_color') ?: '#111';
    $config['shadow'] = (bool) $profile->get('shadow');
    $config['remove_scrollbar'] = (bool) $profile->get('remove_scrollbar');

    // Animation and timing.
    $config['open_close_time'] = (int) $profile->get('open_close_time') ?: 500;
    $config['fading_time'] = (int) $profile->get('fading_time') ?: 150;
    $config['animation_time'] = (int) $profile->get('animation_time') ?: 300;

    // Slideshow options.
    $config['slideshow'] = (bool) $profile->get('slideshow');
    $config['autoplay'] = (bool) $profile->get('autoplay');
    $config['slideshow_time'] = (int) $profile->get('slideshow_time') ?: 6000;
    $config['counter'] = (bool) $profile->get('counter');
    $config['progressbar'] = (bool) $profile->get('progressbar');
    $config['carousel'] = (bool) $profile->get('carousel');

    // Content display options.
    $config['show_title'] = (bool) $profile->get('show_title');
    $config['show_descr'] = (bool) $profile->get('show_descr');
    $config['show_author'] = (bool) $profile->get('show_author');
    $config['txt_toggle_cmd'] = (bool) $profile->get('txt_toggle_cmd');

    // Thumbnails options.
    $config['thumbs_nav'] = (bool) $profile->get('thumbs_nav');
    $config['tn_icons'] = (bool) $profile->get('tn_icons');
    $config['thumbs_w'] = (int) $profile->get('thumbs_w') ?: 110;
    $config['thumbs_h'] = (int) $profile->get('thumbs_h') ?: 110;

    // Fullscreen options.
    $config['fullscreen'] = (bool) $profile->get('fullscreen');
    $config['fs_img_behavior'] = $profile->get('fs_img_behavior') ?: 'fit';
    $config['browser_fs_mode'] = (bool) $profile->get('browser_fs_mode');

    // Social and interaction options.
    $config['socials'] = (bool) $profile->get('socials');
    $config['download'] = (bool) $profile->get('download');
    $config['touchswipe'] = (bool) $profile->get('touchswipe');
    $config['mousewheel'] = (bool) $profile->get('mousewheel');
    $config['autoplay_videos'] = (bool) $profile->get('autoplay_videos');
    $config['rclick_prevent'] = (bool) $profile->get('rclick_prevent');

    // Media options.
    $config['img_zoom'] = (bool) $profile->get('img_zoom');

    // Handle threshold values (can be integer or false).
    $txt_hidden = $profile->get('txt_hidden');
    $config['txt_hidden'] = is_numeric($txt_hidden) ? (int) $txt_hidden : false;

    $tn_hidden = $profile->get('tn_hidden');
    $config['tn_hidden'] = is_numeric($tn_hidden) ? (int) $tn_hidden : false;

    $fs_only = $profile->get('fs_only');
    $config['fs_only'] = is_numeric($fs_only) ? (int) $fs_only : false;

    return $config;
  }

}
