<?php

namespace Drupal\ldap_auth\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\ldap_auth\Utilities;
use Drupal\Core\Url;
use Drupal\ldap_auth\MiniorangeLDAPConstants;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Ajax\RedirectCommand;

class GetStarted extends LDAPFormBase
{
  public function getFormId() {
    return 'ldap_auth_get_started';
  }

  public function buildForm(array $form, FormStateInterface $form_state){

    $form['#attached']['library'][] = 'ldap_auth/ldap_auth.admin';
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';


    $form['markup_14'] = [
      '#markup' => '<div class="mo_ldap_table_layout_1"><div class="mo_ldap_table_layout container">',
    ];

    $form['miniorange_ldap_nltm_desc'] = [
      '#markup' => t('<div>
              <h1>What does miniOrange LDAP module do?</h1><hr>
              <p>The miniOrange Active Directory / LDAP Integration module lets users authenticate in Drupal using their LDAP or Active Directory credentials. It supports secure connection options, flexible attribute mapping, auto user provisioning, and detailed logs to help you manage access with confidence.</p>')
    ];

    // Setup guidance and quick actions.
    $form['setup_info'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['container-inline'],
      ],
      'text' => [
        '#markup' => t('<div><h3>Get started with configuration</h3><p>You can begin by connecting to your own LDAP/AD server, or try a quick demo using a pre-filled dummy server. You can change settings anytime.</p></div>'),
      ],
    ];

    $form['setup_actions'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['action-container'],
      ],
    ];
    $form['setup_actions']['go_config'] = [
      '#type' => 'submit',
      '#value' => t('Configure LDAP server'),
      '#submit' => ['::checkServerConfiguration'],
      '#attributes' => [
        'class' => ['button','button--primary', 'use-ajax'],
      ],
      '#ajax' => [
        'callback' => '::checkServerConfiguration',
        'event' => 'click',
      ],
    ];

    $form['setup_actions']['use_dummy'] = [
      '#type' => 'submit',
      '#id' => 'miniorange_ldap_test_server_button',
      '#value' => $this->t('Configure Dummy LDAP Server'),
      '#submit' => ['::checkServerConfiguration'],
      '#attributes' => [
        'class' => ['button','button--primary', 'use-ajax'],
        'title' => $this->t('Click this button to automatically configure the module using test LDAP server (dummy) details. This allows you to quickly test the module with LDAP attributes without manual setup.'),
      ],  
      '#ajax' => [
        'callback' => '::checkServerConfiguration',
        'event' => 'click',
      ],
    ];

    $features = [
      [
        'title' => t('LDAP Configuration'),
        'desc' => t('Connect to your LDAP/AD server and test instantly.'),
        'route' => 'ldap_auth.ldap_config',
      ],
      [
        'title' => t('NTLM/Kerberos SSO'),
        'desc' => t('Enable Windows Auto Login for seamless SSO.'),
        'route' => 'ldap_auth.signin_settings',
      ],
      [
        'title' => t('Attribute Mapping'),
        'desc' => t('Map LDAP attributes to Drupal user fields.'),
        'route' => 'ldap_auth.attribute_mapping',
      ],
      [
        'title' => t('LDAP Provisioning'),
        'desc' => t('Sync Drupal users to LDAP and import users from LDAP into Drupal.'),
        'route' => 'ldap_auth.user_sync',
      ],
      [
        'title' => t('Advanced Settings'),
        'desc' => t('Fine‑tune login, security and UX options.'),
        'route' => 'ldap_auth.settings',
      ],
      [
        'title' => t('Logs & Reports'),
        'desc' => t('Debug and view audit logs for LDAP activity.'),
        'route' => 'ldap_auth.troubleshoot',
      ],
    ];

    // Use Twig template to render clickable feature boxes.
    $form['grid'] = [
      '#theme' => 'ldap_auth_get_started_cards',
      '#features' => $features,
    ];

    $form['register_close'] = [
      '#markup' => '</div></div>',
    ];

    Utilities::addSupportButton( $form, $form_state);

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state) {

  }

  public function testServerConfiguration(array &$form, FormStateInterface $form_state)
  {
    $this->config_factory->set('miniorange_ldap_protocol', MiniorangeLDAPConstants::TEST_LDAP_PROTOCOL)->save();
    $this->config_factory->set('miniorange_ldap_server_address', MiniorangeLDAPConstants::TEST_SERVER_ADDRESS)->save();
    $this->config_factory->set('miniorange_ldap_server_port_number', MiniorangeLDAPConstants::TEST_SERVER_PORT_NO )->save();
    $this->config_factory->set('miniorange_ldap_server_account_username', MiniorangeLDAPConstants::TEST_SERVER_ACCOUNT_USERNAME)->save();
    $this->config_factory->set('miniorange_ldap_search_base', MiniorangeLDAPConstants::TEST_SEARCH_BASE)->save();
    $this->config_factory->set('miniorange_ldap_username_attribute_option', MiniorangeLDAPConstants::TEST_SEARCH_FILTER)->save();
    $this->config_factory->set('miniorange_ldap_email_attribute', MiniorangeLDAPConstants::TEST_EMAIL_ATTRIBUTE)->save();
    $this->config_factory->set('miniorange_ldap_enable_ldap', MiniorangeLDAPConstants::TEST_ENABLE_LDAP)->save();
    $this->config_factory->set('mo_last_authenticated_user', MiniorangeLDAPConstants::TEST_AUTHENTICATED_USERNAME)->save();

    $msg = $this->t("One-Click Test Configuration:<br>
      &nbsp;&nbsp;- We have set up a <strong>test LDAP server (dummy)</strong> for you to easily test the module.<br>
      &nbsp;&nbsp;- Click on the <strong>\"Test Connection & Proceed\"</strong> button to verify the connection to the LDAP server.<br>
      &nbsp;&nbsp;- Follow this guide to get test server details: <a href='https://www.forumsys.com/2022/05/10/online-ldap-test-server/' target='_blank'>Online LDAP Test Server</a>.<br>"
    );

    $this->messenger->addMessage($msg);
    $form_state->setRedirect('ldap_auth.ldap_config');
  }

  public function checkServerConfiguration(array &$form, FormStateInterface $form_state){
    $is_configured = $this->config->get('miniorange_ldap_is_configured');
    
    if($is_configured){
      // Show the popup window
      return $this->showLDAPServerPopup();
    }
    else{
      // Check if this is the dummy server button
      $triggering_element = $form_state->getTriggeringElement();
      $is_dummy_server = ($triggering_element['#id'] === 'miniorange_ldap_test_server_button');
      
      if($is_dummy_server){
        // Call testServerConfiguration first, then redirect
        $this->testServerConfiguration($form, $form_state);
      }
      
      // Redirect the user to configure the ldap server using AJAX
      $response = new AjaxResponse();
      $redirect_url = Url::fromRoute('ldap_auth.ldap_config')->toString();
      $response->addCommand(new RedirectCommand($redirect_url));
      return $response;
    }
  }

  /**
   * Show popup window for LDAP server configuration options.
   */
  private function showLDAPServerPopup() {
    $response = new AjaxResponse();
    
    $server_name = $this->config->get('miniorange_ldap_server');
    $server_address = $this->config->get('miniorange_ldap_server_address');
    $service_account = $this->config->get('miniorange_ldap_server_account_username');
    
    // Determine server info to display
    $server_info = !empty($server_name) ? $server_name : (!empty($server_address) ? $server_address : 'Unknown');
    $account_info = !empty($service_account) ? $service_account : 'No Account Found';
    
          $message = $this->t('<div class="ldap-server-popup">
        <h3>LDAP Server Configuration Limit</h3>
        <div class="ldap-message">
          <p><strong>You can add only 1 LDAP server in this module.</strong></p>
          <p>If you wish to configure multiple LDAP Servers, then upgrade your plan to add more servers.</p>
        </div>
        <div class="ldap-current-config">
          <h4>Current Configuration:</h4>
          <ul>
            <li><strong>Server:</strong> @server</li>
            <li><strong>Service Account:</strong> @account</li>
          </ul>
        </div>
        <div class="ldap-options">
          <a href="@upgrade_url" class="button button--primary" target="_blank">Upgrade Plan</a>
          <p class="or-text">OR</p>
          <p class="delete-text">Delete the existing one to add new configuration:</p>
          <a href="@delete_url" class="button button--danger">Delete Configuration</a>
        </div>
      </div>', [
        '@server' => $server_info,
        '@account' => $account_info,
        '@upgrade_url' => MiniorangeLDAPConstants::All_INCLUSIVE_PLAN,
        '@delete_url' => Url::fromRoute('ldap_auth.delete_config', ['action' => 'delete'])->toString(),
      ]);
    
    // Add modal dialog command
    $response->addCommand(new OpenModalDialogCommand(
      t('LDAP Server Configuration'),
      $message,
      ['width' => '600px', 'height' => 'auto']
    ));
    
    return $response;
  }



}
