<?php

namespace Drupal\ldap_auth;

use Drupal\user\UserInterface;

/**
 * Handles Drupal → LDAP user synchronization.
 */
class UserSyncService {

  private ?LDAPFlow $ldap = null;
  private $conn = null;
  private bool $isBound = false;
  private $config;

  private function getConfig() {
    if (!$this->config) {
      $this->config = \Drupal::config('ldap_auth.settings');
    }
    return $this->config;
  }

  private function getLDAPConnection() {
    if ($this->isBound && $this->conn) {
      return $this->conn;
    }
    if (!Utilities::isLDAPInstalled() || !Utilities::isOpenSSLEnabled()) {
      return null;
    }
    if ($this->ldap === null) {
      $this->ldap = new LDAPFlow();
    }
    $this->conn = $this->ldap->getConnection();
    if (!$this->conn) {
      return null;
    }
    $bind = @ldap_bind($this->conn, $this->ldap->getServiceAccountUsername(), $this->ldap->getServiceAccountPassword());
    if (!$bind) {
      return null;
    }
    $this->isBound = true;
    return $this->conn;
  }

  /**
   * Create a corresponding LDAP user when a Drupal user is created.
   */
  public function syncOnUserInsert(UserInterface $account): void {
    $config = $this->getConfig();
    if (!$config->get('create_user_in_ldap')) {
      return;
    }

    $conn = $this->getLDAPConnection();
    if (!$conn) {
      LDAPLOGGERS::addLogger('US1: Unable to bind to LDAP', '', __LINE__, __FUNCTION__, __FILE__);
      return;
    }

    $usernameAttr = $config->get('miniorange_ldap_username_attribute') ?: 'samaccountname';
    $username = $account->getAccountName();
    $filter = '(&(' . $usernameAttr . '=' . ldap_escape($username, '', LDAP_ESCAPE_FILTER) . ')(|(objectClass=user)(objectClass=person)(objectClass=inetOrgPerson)))';
    $searchBase = $config->get('miniorange_ldap_search_base');

    // Check if user already exists in LDAP.
    $search = @ldap_search($conn, $searchBase, $filter, ['dn']);
    if ($search) {
      $entries = @ldap_get_entries($conn, $search);
      if ($entries && $entries['count'] > 0) {
        return;
      }
    }
    $dnAttr = $config->get('miniorange_ldap_user_dn_format') ?: 'cn';
    $dnValue = ldap_escape($username, '', LDAP_ESCAPE_DN);
    $dn = $dnAttr . '=' . $dnValue . ',' . $searchBase;

    $emailAttr = $config->get('miniorange_ldap_email_attribute') ?: 'mail';
    $mail = $account->getEmail();
    $commonName = $account->getDisplayName() ?: $username;
    $surname = $username;

    $entry = [
      'cn' => [$commonName],
      'sn' => [$surname],
    ];
    $entry[$usernameAttr] = [$username];
    if (!empty($mail)) {
      $entry[$emailAttr] = [$mail];
    }
    $entry['objectClass'] = ['top', 'person', 'organizationalPerson', 'inetOrgPerson'];
    LDAPLOGGERS::addLogger('US1: Attempting LDAP user creation', $username, __LINE__, __FUNCTION__, __FILE__);
    $added = @ldap_add($conn, $dn, $entry);
    if (!$added) {
      $entry['objectClass'] = ['top', 'person', 'organizationalPerson', 'user'];
      $added = @ldap_add($conn, $dn, $entry);
    }

    if (!$added) {
      $errors = Utilities::getLDAPDiagnosticError($conn);
      LDAPLOGGERS::addLogger('US2: LDAP add failed', $errors, __LINE__, __FUNCTION__, __FILE__);
    } else {
      LDAPLOGGERS::addLogger('US3: LDAP user created successfully', $username, __LINE__, __FUNCTION__, __FILE__);
    }
  }

  /**
   * Update LDAP attributes when a Drupal user is updated.
   *
   * @param UserInterface $account
   * @param UserInterface|null $original
   */
  public function syncOnUserUpdate(UserInterface $account, ?UserInterface $original = null): void {
    $config = $this->getConfig();
    if (!$config->get('miniorange_ldap_update_user_info')) {
      return;
    }

    // If original provided and no relevant changes, skip.
    if ($original instanceof UserInterface) {
      $emailChanged = $original->getEmail() !== $account->getEmail();
      $nameChanged = $original->getDisplayName() !== $account->getDisplayName();
      if (!$emailChanged && !$nameChanged) {
        return;
      }
    }

    $conn = $this->getLDAPConnection();
    if (!$conn) {
      LDAPLOGGERS::addLogger('US3: Unable to bind to LDAP', '', __LINE__, __FUNCTION__, __FILE__);
      return;
    }

    $usernameAttr = $config->get('miniorange_ldap_username_attribute') ?: 'samaccountname';
    $username = $account->getAccountName();
    $searchBase = $config->get('miniorange_ldap_search_base');
    $emailAttr = $config->get('miniorange_ldap_email_attribute') ?: 'mail';

    $filter = '(&(' . $usernameAttr . '=' . ldap_escape($username, '', LDAP_ESCAPE_FILTER) . ')(|(objectClass=user)(objectClass=person)(objectClass=inetOrgPerson)))';
    $attrs = ['dn', $emailAttr, 'cn', 'sn'];
    $search = @ldap_search($conn, $searchBase, $filter, $attrs);
    if (!$search) {
      $errors = Utilities::getLDAPDiagnosticError($conn);
      LDAPLOGGERS::addLogger('US4: LDAP search failed', $errors, __LINE__, __FUNCTION__, __FILE__);
      return;
    }
    $entries = @ldap_get_entries($conn, $search);
    if (!$entries || $entries['count'] === 0) {
      return;
    }
    $dn = $entries[0]['dn'];

    $mods = [];
    $newEmail = (string) $account->getEmail();
    $newCn = (string) ($account->getDisplayName() ?: $username);
    $newSn = (string) $username;

    $curEmail = '';
    if (isset($entries[0][strtolower($emailAttr)]['count']) && $entries[0][strtolower($emailAttr)]['count'] > 0) {
      $curEmail = (string) $entries[0][strtolower($emailAttr)][0];
    }
    $curCn = isset($entries[0]['cn']['count']) && $entries[0]['cn']['count'] > 0 ? (string) $entries[0]['cn'][0] : '';
    $curSn = isset($entries[0]['sn']['count']) && $entries[0]['sn']['count'] > 0 ? (string) $entries[0]['sn'][0] : '';

    if ($newEmail !== '' && $newEmail !== $curEmail) {
      $mods[$emailAttr] = [$newEmail];
    }
    if ($newCn !== '' && $newCn !== $curCn) {
      $mods['cn'] = [$newCn];
    }
    if ($newSn !== '' && $newSn !== $curSn) {
      $mods['sn'] = [$newSn];
    }

    if (!empty($mods)) {
      LDAPLOGGERS::addLogger('US5: Attempting LDAP user update', $username, __LINE__, __FUNCTION__, __FILE__);
      $result = @ldap_modify($conn, $dn, $mods);
      if ($result) {
        LDAPLOGGERS::addLogger('US6: LDAP user updated successfully', $username, __LINE__, __FUNCTION__, __FILE__);
      } else {
        $errors = Utilities::getLDAPDiagnosticError($conn);
        LDAPLOGGERS::addLogger('US7: LDAP user update failed', $errors, __LINE__, __FUNCTION__, __FILE__);
      }
    }
  }

  /**
   * Delete LDAP user when the Drupal user is deleted.
   */
  public function syncOnUserDelete(UserInterface $account): void {
    $config = $this->getConfig();

    if (!$config->get('create_user_in_ldap')) {
      return;
    }

    $conn = $this->getLDAPConnection();
    if (!$conn) {
      LDAPLOGGERS::addLogger('USD1: Unable to bind to LDAP', '', __LINE__, __FUNCTION__, __FILE__);
      return;
    }

    $usernameAttr = $config->get('miniorange_ldap_username_attribute') ?: 'samaccountname';
    $username = $account->getAccountName();
    $searchBase = $config->get('miniorange_ldap_search_base');
    $filter = '(&(' . $usernameAttr . '=' . ldap_escape($username, '', LDAP_ESCAPE_FILTER) . ')(|(objectClass=user)(objectClass=person)(objectClass=inetOrgPerson)))';

    $search = @ldap_search($conn, $searchBase, $filter, ['dn']);
    if (!$search) {
      $errors = Utilities::getLDAPDiagnosticError($conn);
      LDAPLOGGERS::addLogger('USD2: LDAP search failed', $errors, __LINE__, __FUNCTION__, __FILE__);
      return;
    }
    $entries = @ldap_get_entries($conn, $search);
    if (!$entries || $entries['count'] === 0) {
      return;
    }

    $dn = $entries[0]['dn'];
    LDAPLOGGERS::addLogger('USD1: Attempting LDAP user deletion', $username, __LINE__, __FUNCTION__, __FILE__);
    $ok = @ldap_delete($conn, $dn);
    if (!$ok) {
      $errors = Utilities::getLDAPDiagnosticError($conn);
      LDAPLOGGERS::addLogger('USD2: LDAP delete failed', $errors, __LINE__, __FUNCTION__, __FILE__);
    } else {
      LDAPLOGGERS::addLogger('USD3: LDAP user deleted successfully', $username, __LINE__, __FUNCTION__, __FILE__);
    }
  }
}


