<?php

namespace Drupal\ldap_auth\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\ldap_auth\Utilities;
use Drupal\Core\Url;
use Drupal\ldap_auth\MiniorangeLDAPConstants;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Ajax\CloseDialogCommand;

class GetStarted extends LDAPFormBase
{
  public function getFormId() {
    return 'ldap_auth_get_started';
  }

  public function buildForm(array $form, FormStateInterface $form_state){
    $form['#attached']['library'][] = 'ldap_auth/ldap_auth.admin';
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';
    $form['#attached']['library'][] = 'core/drupal.ajax';
    $form['#attached']['library'][] = 'ldap_auth/ldap_auth.notify';
    $form['#attached']['drupalSettings']['ldap_auth']['notifyUsUrl'] =
      Url::fromRoute('ldap_auth.notify_us_popup', [], ['query' => ['action' => 'open']])->toString();
    $form['#attached']['library'][] = 'ldap_auth/ldap_auth.mo_ldap_tooltip';

    $query_parameter = \Drupal::request()->query->get('action');
    if ($query_parameter === 'delete') {
      $this->showDeleteConfirmationForm($form, $form_state);
      return $form;
    }


    $form['markup_14'] = [
      '#markup' => '<div class="mo_ldap_table_layout_1"><div class="mo_ldap_table_layout container">',
    ];

    $form['miniorange_ldap_nltm_desc'] = [
      '#markup' => t('<div class="mo-gs-hero">
              <h1>Welcome to your LDAP Integration Dashboard</h1>
              <p>Quickly connect your directory service and manage user access securely.</p>
            </div>')
    ];

    $form['setup_actions'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['action-container', 'mo-gs-actions'],
      ],
    ];

    // Left action: Configure LDAP server + description.
    $form['setup_actions']['primary'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['mo-gs-action']],
    ];
    $form['setup_actions']['primary']['go_config'] = [
      '#type' => 'submit',
      '#value' => t('Configure LDAP server'),
      '#submit' => ['::checkServerConfiguration'],
      '#attributes' => [
        'class' => ['button','button--primary', 'use-ajax', 'mo-gs-btn', 'mo-gs-btn--primary', 'mo-gs-btn--withicon', 'mo-gs-btn--server'],
      ],
      '#ajax' => [
        'callback' => '::checkServerConfiguration',
        'event' => 'click',
      ],
    ];
    $form['setup_actions']['primary']['go_desc'] = [
      '#markup' => '<div class="mo-gs-btn-desc">Connect to your live Active Directory/LDAP.</div>',
    ];
    $form['setup_actions']['secondary'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['mo-gs-action']],
    ];
    $form['setup_actions']['secondary']['use_dummy'] = [
      '#type' => 'submit',
      '#id' => 'miniorange_ldap_test_server_button',
      '#value' => $this->t('Configure Dummy LDAP Server'),
      '#submit' => ['::checkServerConfiguration'],
      '#attributes' => [
        'class' => ['button', 'use-ajax', 'mo-gs-btn', 'mo-gs-btn--outline', 'mo-gs-btn--withicon', 'mo-gs-btn--flask'],
      ],
      '#ajax' => [
        'callback' => '::checkServerConfiguration',
        'event' => 'click',
      ],
      '#prefix' => '<div class="mo-btn-wrapper">',
      '#suffix' => $this->getTooltipMarkup(
        t('Configure Dummy LDAP Server'),
        t('This button auto-fills configuration using the Online LDAP Test Server details from <a href="https://www.forumsys.com/2022/05/10/online-ldap-test-server/" target="_blank" rel="noopener noreferrer">Forum Systems</a> so you can immediately test other module features without a live LDAP.')
      ) . '</div>',
    ];
    $form['setup_actions']['secondary']['dummy_desc'] = [
      '#markup' => '<div class="mo-gs-btn-desc">Don\'t have details yet? Test with sample data.</div>',
    ];


    $features = [
      [
        'title' => t('LDAP Configuration'),
        'desc' => t('Connect to your LDAP/AD server and test instantly.'),
        'route' => 'ldap_auth.ldap_config',
      ],
      [
        'title' => t('NTLM/Kerberos SSO'),
        'desc' => t('Enable Windows Auto Login for seamless SSO.'),
        'route' => 'ldap_auth.signin_settings',
      ],
      [
        'title' => t('Attribute Mapping'),
        'desc' => t('Map LDAP attributes to Drupal user fields.'),
        'route' => 'ldap_auth.attribute_mapping',
      ],
      [
        'title' => t('LDAP Provisioning'),
        'desc' => t('Sync Drupal users to LDAP and import users from LDAP into Drupal.'),
        'route' => 'ldap_auth.user_sync',
      ],
      [
        'title' => t('Advanced Settings'),
        'desc' => t('Fine‑tune login, security and UX options.'),
        'route' => 'ldap_auth.settings',
      ],
      [
        'title' => t('Logs & Reports'),
        'desc' => t('Debug and view audit logs for LDAP activity.'),
        'route' => 'ldap_auth.troubleshoot',
      ],
    ];
    $features_markup = '<div class="mo-gs-grid">';
    $icon_classes = [
      'config' => 'mo-gs-icon--config',
      'ntlm' => 'mo-gs-icon--ntlm',
      'mapping' => 'mo-gs-icon--mapping',
      'prov' => 'mo-gs-icon--prov',
      'advanced' => 'mo-gs-icon--advanced',
      'logs' => 'mo-gs-icon--logs',
    ];

    $cards = [
      ['title' => $this->t('LDAP Configuration'), 'desc' => $this->t('Connect to your LDAP/AD server and test instantly.'), 'route' => 'ldap_auth.ldap_config', 'icon' => $icon_classes['config']],
      ['title' => $this->t('NTLM/Kerberos SSO'), 'desc' => $this->t('Enable Windows Auto Login for seamless SSO.'), 'route' => 'ldap_auth.signin_settings', 'icon' => $icon_classes['ntlm']],
      ['title' => $this->t('Attribute Mapping'), 'desc' => $this->t('Map LDAP attributes to Drupal user fields.'), 'route' => 'ldap_auth.attribute_mapping', 'icon' => $icon_classes['mapping']],
      ['title' => $this->t('LDAP Provisioning'), 'desc' => $this->t('Sync Drupal users to LDAP and import users from LDAP into Drupal.'), 'route' => 'ldap_auth.user_sync', 'icon' => $icon_classes['prov']],
      ['title' => $this->t('Advanced Settings'), 'desc' => $this->t('Fine‑tune login, security and UX options.'), 'route' => 'ldap_auth.settings', 'icon' => $icon_classes['advanced']],
      ['title' => $this->t('Logs & Reports'), 'desc' => $this->t('Debug and view audit logs for LDAP activity.'), 'route' => 'ldap_auth.troubleshoot', 'icon' => $icon_classes['logs']],
    ];

    foreach ($cards as $card) {
      $url = Url::fromRoute($card['route'])->toString();
      $highlight = !empty($card['highlight']) ? ' mo-gs-card--selected' : '';
      $features_markup .= '<a class="mo-gs-card' . $highlight . '" href="' . $url . '">
          <span class="mo-gs-card__icon ' . $card['icon'] . '"></span>
          <span class="mo-gs-card__title">' . $card['title'] . '</span>
          <span class="mo-gs-card__desc">' . $card['desc'] . '</span>
        </a>';
    }
    $features_markup .= '</div>';

    $form['features_v2'] = [
      '#markup' => $features_markup,
    ];

    $form['register_close'] = [
      '#markup' => '</div></div>',
    ];

    Utilities::addSupportButton( $form, $form_state);

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state) {

  }

  public function testServerConfiguration(array &$form, FormStateInterface $form_state)
  {
    $this->config_factory->set('miniorange_ldap_protocol', MiniorangeLDAPConstants::TEST_LDAP_PROTOCOL)->save();
    $this->config_factory->set('miniorange_ldap_server_address', MiniorangeLDAPConstants::TEST_SERVER_ADDRESS)->save();
    $this->config_factory->set('miniorange_ldap_server_port_number', MiniorangeLDAPConstants::TEST_SERVER_PORT_NO )->save();
    $this->config_factory->set('miniorange_ldap_server_account_username', MiniorangeLDAPConstants::TEST_SERVER_ACCOUNT_USERNAME)->save();
    $this->config_factory->set('miniorange_ldap_search_base', MiniorangeLDAPConstants::TEST_SEARCH_BASE)->save();
    $this->config_factory->set('miniorange_ldap_username_attribute_option', MiniorangeLDAPConstants::TEST_SEARCH_FILTER)->save();
    $this->config_factory->set('miniorange_ldap_email_attribute', MiniorangeLDAPConstants::TEST_EMAIL_ATTRIBUTE)->save();
    $this->config_factory->set('miniorange_ldap_enable_ldap', MiniorangeLDAPConstants::TEST_ENABLE_LDAP)->save();
    $this->config_factory->set('mo_last_authenticated_user', MiniorangeLDAPConstants::TEST_AUTHENTICATED_USERNAME)->save();

    $msg = $this->t("One-Click Test Configuration:<br>
      &nbsp;&nbsp;- We have set up a <strong>test LDAP server (dummy)</strong> for you to easily test the module.<br>
      &nbsp;&nbsp;- Click on the <strong>\"Test Connection & Proceed\"</strong> button to verify the connection to the LDAP server.<br>
      &nbsp;&nbsp;- Follow this guide to get test server details: <a href='https://www.forumsys.com/2022/05/10/online-ldap-test-server/' target='_blank'>Online LDAP Test Server</a>.<br>"
    );

    $this->messenger->addMessage($msg);
    $form_state->setRedirect('ldap_auth.ldap_config');
  }

  public function checkServerConfiguration(array &$form, FormStateInterface $form_state){
    $is_configured = $this->config->get('miniorange_ldap_is_configured');

    if($is_configured){
      // Show the popup window
      return $this->showLDAPServerPopup();
    }
    else{
      // Check if this is the dummy server button
      $triggering_element = $form_state->getTriggeringElement();
      $is_dummy_server = ($triggering_element['#id'] === 'miniorange_ldap_test_server_button');

      if($is_dummy_server){
        // Call testServerConfiguration first, then redirect
        $this->testServerConfiguration($form, $form_state);
      }

      // Redirect the user to configure the ldap server using AJAX
      $response = new AjaxResponse();
      $redirect_url = Url::fromRoute('ldap_auth.ldap_config')->toString();
      $response->addCommand(new RedirectCommand($redirect_url));
      return $response;
    }
  }

  /**
   * Show popup window for LDAP server configuration options.
   */
  private function showLDAPServerPopup() {
    $response = new AjaxResponse();

    $server_name = $this->config->get('miniorange_ldap_server');
    $server_address = $this->config->get('miniorange_ldap_server_address');
    $service_account = $this->config->get('miniorange_ldap_server_account_username');

    // Determine server info to display
    $server_info = !empty($server_name) ? $server_name : (!empty($server_address) ? $server_address : 'Unknown');
    $account_info = !empty($service_account) ? $service_account : 'No Account Found';

          $message = $this->t('<div class="ldap-server-popup">
        <h3>LDAP Server Configuration Limit</h3>
        <div class="ldap-message">
          <p>Your plan supports configuring one LDAP server, to add additional LDAP servers please upgrade your plan</p>
        </div>
        <div class="ldap-current-config">
          <h4>Current Configuration:</h4>
          <ul>
            <li><strong>Server:</strong> @server</li>
            <li><strong>Service Account:</strong> @account</li>
          </ul>
        </div>
        <div class="ldap-options">
          <a href="@upgrade_url" class="button button--primary" target="_blank">Upgrade Plan</a>
          <p class="or-text">OR</p>
          <p class="delete-text">Delete the existing one to add new configuration:</p>
          <a href="@delete_url" class="button button--danger use-ajax" data-dialog-type="modal" data-dialog-options="{&quot;width&quot;:&quot;40%&quot;, &quot;title&quot;:&quot;Delete LDAP Configuration&quot;}">Delete Configuration</a>
        </div>
      </div>', [
        '@server' => $server_info,
        '@account' => $account_info,
        '@upgrade_url' => MiniorangeLDAPConstants::All_INCLUSIVE_PLAN,
        '@delete_url' => Url::fromRoute('ldap_auth.get_started', [], ['query' => ['action' => 'delete']])->toString(),
      ]);

    // Add modal dialog command
    $response->addCommand(new OpenModalDialogCommand(
      t('LDAP Server Configuration'),
      $message,
      ['width' => '600px', 'height' => 'auto']
    ));

    return $response;
  }

  protected function showDeleteConfirmationForm(array &$form, FormStateInterface $form_state) {
    $this->addDeleteConfirmationForm($form, $form_state, 'resetConfigurationGetStarted');
  }

  public function resetConfigurationGetStarted($form, $form_state) {
    $this->clearLdapConfigurationValues();

    Utilities::add_message($this->t('Configurations removed successfully.'), 'status');
    
    if (\Drupal::request()->isXmlHttpRequest()) {
      $response = new AjaxResponse();
      $response->addCommand(new CloseDialogCommand());
      $response->addCommand(new RedirectCommand(Url::fromRoute('ldap_auth.get_started')->toString()));
      $form_state->setResponse($response);
      return;
    }
    $form_state->setRedirect('ldap_auth.get_started');
  }

  private function getTooltipMarkup($title, $description) {
    $helper_text = '<div class="mo-ldap--help--content">' . $description . '</div>';
    $helper_text = htmlspecialchars($helper_text, ENT_QUOTES, 'UTF-8');
    return '<span role="tooltip" tabindex="0" aria-expanded="false" class="mo-ldap--help js-miniorange-ldap-help miniorange-ldap-help" data-miniorange-ldap-help="' . $helper_text . '"><span aria-hidden="true">i</span></span>';
  }

}
