<?php

namespace Drupal\ldap_auth\Form;

use Drupal\Component\Utility\EmailValidatorInterface;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Url;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 *
 */
class LDAPFormBase extends FormBase {

  /**
   * The base URL of the Drupal installation.
   */
  protected $entityTypeManager;
  protected string $base_url;

  /**
   * A config object fetching configuration in config table.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected Config $config;

  /**
   * A config object for storing, updating, and deleting stored configuration in config table.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected Config $config_factory;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The logger factory.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  protected Request $request;

  protected bool $disabled;

  /**
   * @var mixed
   */
  /**
   * @var \Drupal\Component\Utility\EmailValidatorInterface
   */
  protected EmailValidatorInterface $emailValidator;

  /**
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $moduleList;

  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory,
  MessengerInterface $messenger,
  LoggerChannelFactoryInterface $logger_factory,
  EmailValidatorInterface $email_validator,
  ModuleExtensionList $module_list,
  Connection $database, EntityTypeManagerInterface $entity_type_manager ) {
    $this->base_url = \Drupal::request()->getSchemeAndHttpHost().\Drupal::request()->getBasePath();
    $this->config = $config_factory->getEditable('ldap_auth.settings');
    $this->config_factory = $config_factory->getEditable('ldap_auth.settings');
    $this->messenger = $messenger;
    $this->logger = $logger_factory->get('ldap_auth');
    $this->request = \Drupal::request();
    $this->emailValidator = $email_validator;
    $this->moduleList = $module_list;
    $this->database = $database;
    $this->disabled = FALSE;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   *
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('messenger'),
      $container->get('logger.factory'),
      $container->get('email.validator'),
      $container->get('extension.list.module'),
      $container->get('database'),
      $container->get('entity_type.manager') 
    );
  }

  /**
   *
   */
  public function getFormId() {
    return 'ldap_form_base';
  }

  /**
   *
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   *
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // @todo Implement submitForm() method.
  }

  public function getRouteUrl($route_name){
    $link = Url::fromRoute($route_name)->toString(TRUE)->getGeneratedUrl();
    return \Drupal::request()->getSchemeAndHttpHost().$link;
  }

  /**
   * Clear all LDAP configuration values to defaults.
   */
  protected function clearLdapConfigurationValues(): void {
    $this->config_factory->clear('miniorange_ldap_enable_ldap')
        ->clear('miniorange_ldap_authenticate_admin')
        ->clear('miniorange_ldap_authenticate_drupal_users')
        ->clear('miniorange_ldap_enable_auto_reg')
        ->clear('miniorange_ldap_server')
        ->clear('miniorange_ldap_server_account_username')
        ->clear('miniorange_ldap_server_account_password')
        ->clear('miniorange_ldap_search_base')
        ->clear('miniorange_ldap_username_attribute')
        ->clear('miniorange_ldap_test_username')
        ->clear('miniorange_ldap_test_password')
        ->clear('miniorange_ldap_server_address')
        ->clear('miniorange_ldap_enable_anony_bind')
        ->clear('miniorange_ldap_protocol')
        ->clear('miniorange_ldap_username_attribute_option')
        ->clear('ldap_binding_options')
        ->clear('miniorange_ldap_is_configured')
        ->clear('possible_ldap_search_bases')
        ->clear('ldap_attribute_list')
        ->clear('ldap_user_attributes_and_values')
        ->save();

    $this->config_factory->set('miniorange_ldap_server_port_number', '389')->save();
    $this->config_factory->set('miniorange_ldap_custom_username_attribute', 'samaccountName')->save();
    $this->config_factory->set('miniorange_ldap_config_status', 'two')->save();
    $this->config_factory->set('miniorange_ldap_steps', "0")->save();
  }

  /**
   * Build a standardized delete confirmation form (message + actions).
   */
  protected function addDeleteConfirmationForm(array &$form, FormStateInterface $form_state, string $submitMethodName): void {
    $form['delete_confirmation_message'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Are you sure you want to delete the configuration? <strong>This action cannot be undone.</strong>') . '</p>',
    ];

    $form['delete_confirmation_actions'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['form-actions'],
        'style' => 'display:flex; gap: 8px; justify-content:flex-start;',
      ],
    ];

    $form['delete_confirmation_actions']['cancel_delete'] = [
      '#type' => 'button',
      '#value' => $this->t('Cancel'),
      '#attributes' => ['class' => ['button', 'dialog-cancel']],
    ];

    $form['delete_confirmation_actions']['confirm_delete'] = [
      '#type' => 'submit',
      '#value' => $this->t('Delete Configuration'),
      '#submit' => ['::' . $submitMethodName],
      '#attributes' => ['class' => ['button', 'button--danger']],
    ];
  }

}
