<?php

namespace Drupal\ldap_auth\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\ldap_auth\MiniorangeLDAPConstants;
use Drupal\ldap_auth\Utilities;
use Drupal\user\Entity\User;
use Drupal\ldap_auth\LDAPLOGGERS;
use Symfony\Component\HttpFoundation\Response;



class MiniorangeAdvanceSetting extends LDAPFormBase {

  public function getFormId()
  {
    return "advance_settings";
  }

  public function buildForm(array $form, FormStateInterface $form_state)
  {

    $upgrade_plan_link = $this->getRouteUrl('ldap_auth.licensing');

    $form['markup_library'] = [
        '#attached' => [
            'library' => [
                "ldap_auth/ldap_auth.admin",
                "core/drupal.dialog.ajax"
            ],
        ],
    ];

    $form['markup_start'] = [
        '#type' => 'markup',
        '#markup' => '<div class="mo_ldap_table_layout_1"><div class="mo_ldap_table_layout container" >',
    ];

    $form['export_import_configuration_section'] = [
      '#type' => 'details',
      '#title' => t('Import / Export Configuration'),
      '#open' => true,
    ];

    $form['export_import_configuration_section']['markup_top'] = [
      '#markup' => '<div class="export-import-configuration-header">
                      <h3>' . $this->t('Export Configuration') . '</h3>
                      <p>' . $this->t('Click on the button below to download the module configuration file. This file can be used for importing configuration into another site.') . '</p>
                    </div>',
    ];


    $form['export_import_configuration_section']['miniorange_ldap_exist_export'] = [
      '#type' => 'submit',
      '#button_type' => 'primary',
      '#value' => t('Download Configuration'),
      '#submit' => ['::exportConfigurationToJson'],
    ];

    $form['export_import_configuration_section']['markup_middle'] = [
      '#markup' => '<hr><div class="import-configuration-header">
                      <h3>' . $this->t('Import Configuration') . '</h3>
                      <p>' . $this->t('Choose a <b>"JSON"</b> module configuration file and upload it by clicking the button below.') . '</p>
                    </div>',
    ];


    $form['export_import_configuration_section']['import_Config_file'] = [
      '#type' => 'file',
      '#attributes'=> ['accept' => 'application/json'],
      '#prefix' => '<div class="container-inline">',
    ];

    $form['export_import_configuration_section']['miniorange_saml_idp_import'] = [
      '#type' => 'submit',
      '#value' => t('Upload'),
      '#button_type' => 'primary',
      '#submit' => ['::importConfiguration'],
      '#suffix' => '</div><br><br>',
    ];

    $form['export_import_configuration_section']['markup_bottom'] = [
      '#markup' => '<div class="export-import-configuration-footer">
                      <p><strong>' . $this->t('Note:') . '</strong> ' .
        $this->t('Ensure you save the downloaded file securely. It contains sensitive configuration details.') . '</p>
                    </div>',
    ];

    // UPDATE LOGIN FORM
    $form['user_login_interface'] = [
      '#type' => 'details',
      '#title'=> $this->t("User Login Interface"),
      '#open' => true,
    ];

    $form['user_login_interface']['username_description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username Description'),
      '#maxlength' => 2048,
      '#description' => $this->t('The above description will override the current Username field description on user login form.'),
      '#default_value' => $this->config->get('username_description')
    ];

    $form['user_login_interface']['password_description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Password Description'),
      '#maxlength' => 2048,
      '#description' => $this->t('The above description will override the current Password field description on user login form.'),
      '#default_value' => $this->config->get('password_description')
    ];


    //Redirect user after login
    $form['redirect'] = [
        "#type" => "details",
        '#title' => $this->t('Custom Login/Logout Redirect URL'),
    ];

    $form['redirect']['info'] = [
        '#type' => 'fieldset',
        '#attributes' => [
            'style' => 'background-color:#f3f3f3;box-shadow:none;border:none;width:80%;',
            'class' => ['ldap-user-sync'],
        ],
    ];

    $form['redirect']['info']['sync_markup_note'] = [
        '#markup' => $this->t('<div><ul style="line-height:150%">
<li><b>Configure the user redirection after user login and logout upon LDAP Authentication.</b></li>
<li>This feature is available in the <a href='.$upgrade_plan_link.' target="_blank"><b>All-Inclusive</b></a> version of the module.</li></ul>
</div>'),
    ];

    $form['redirect']['miniorange_ldap_login_redirect'] = [
        '#type' => 'url',
        '#title' => t('Redirect URL After Login:'),
        '#disabled' => true,
        '#description' => $this->t('<strong>Note: </strong>Enter the entire URL (<em> including https:// </em>) where you want to redirect user after successful authentication.'),
        '#attributes' => ['placeholder' => 'Eg. https://www.example.com'],
    ];

    $form['redirect']['miniorange_ldap_logout_redirect'] = [
        '#type' => 'url',
        '#title' => t('Redirect URL After Logout:'),
        '#disabled' => true,
        '#description' => t('<strong>Note: </strong>Enter the entire URL (<em> including https:// </em>) where you want to redirect user after logout.'),
        '#attributes' => ['placeholder' => 'Eg. https://www.example.com'],
    ];

   // DISABLE USER PROFILE FIELDS

    $form['miniorange_ldap_auto_disable_fieldset'] = [
        '#type' => 'details',
        '#title' => $this->t('Disable User Profile Fields'),
    ];

    $form['miniorange_ldap_auto_disable_fieldset']['info'] = [
        '#type' => 'fieldset',
        '#attributes' => [
            'style' => 'background-color:#f3f3f3;box-shadow:none;border:none;width:80%;',
            'class' => ['ldap-user-sync'],
        ],
    ];

    $form['miniorange_ldap_auto_disable_fieldset']['info']['sync_markup_note'] = [
        '#markup' => $this->t('<div><ul style="line-height:150%">
<li>Disable or hide user profile fields in the user form. <i>eg registration form, custom_user form</i></li>
<li>Prevent user password updates by disabling or hiding the password field in the user profile edit form or any other custom form.</li>
<li>This feature is available in the <a href='.$upgrade_plan_link.' target="_blank"><b>All-Inclusive</b></a> version of the module.</li></ul>
</div>'),
    ];

    $form['miniorange_ldap_auto_disable_fieldset']['miniorange_ldap_disable_profile_field'] = [
        '#type' => 'checkbox',
        '#disabled' => true,
        '#title' => t('<b>Enable this checkbox to disable users profile attribute fields.</b>'),
    ];

    $form['miniorange_ldap_auto_disable_fieldset']['miniorange_ldap_disable_user_profile_attributes'] = [
        '#type' => 'select',
        '#title' => t('Select profile attributes to disable'),
        '#multiple' => TRUE,
        '#options' =>  $this->CustomFieldStorage(),
        '#default_value' => $this->config->get('miniorange_ldap_disable_user_profile_attributes'),
        '#description' => '<b>Note:</b> The users would not be able to change these attributes.',
        '#attributes' => ['placeholder' => "Select profile attributes to disable"],
    ];

    $form['miniorange_ldap_auto_disable_fieldset']['miniorange_ldap_disable_pass_confirm_pass'] = [
        '#title' => t('Disable/Hide "<u>Current password</u>", "<u>Password</u>" and "<u>Current Password</u>" fields of user profile page:'),
        '#type' => 'radios',
        '#options' => [
            'editable' => t('Keep Editable - Users can see and update their passwords.'),
            'disable' => t('Disable - Users can see the password fields but can’t edit them.'),
            'hide' => t('Disable and Hide - Hide password fields on all user forms except the login form.'),
        ],
        '#disabled' => TRUE,
    ];


    $form['ldap_group_restriction'] = [
        '#type' => 'details',
        '#title'=> $this->t("Restrict LDAP Groups"),
        '#disabled' => true,
    ];

    $form['ldap_group_restriction']['info'] = [
        '#type' => 'fieldset',
        '#attributes' => [
            'style' => 'background-color:#f3f3f3;box-shadow:none;border:none;width:80%;',
            'class' => ['ldap-user-sync'],
        ],
    ];

    $form['ldap_group_restriction']['info']['sync_markup_note'] = [
        '#markup' => $this->t('<div><ul style="line-height:150%">
<li>Configure LDAP Group Based Login Restriction in this section.</li>
<li>Allow or block user logins based on listed LDAP groups.</li>
<li>This feature is available in the <a href='.$upgrade_plan_link.' target="_blank"><b>All-Inclusive</b></a> version of the module.</li></ul>
<a  class="button button--primary button--small" href='.MiniorangeLDAPConstants::GROUP_LOGIN_RESTRICTION_GUIDE.' target="_blank">🕮 Setup guide</a>
</div>'),
    ];

    $form['ldap_group_restriction']['group_restriction'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable Allowing and Blocking LDAP Groups'),
    ];

    $form['ldap_group_restriction']['whitelist_blacklist_radio_buttons'] = [
        '#type' => 'radios',
        '#title' => $this->t('Select restriction option'),
        '#default_value' => $this->config->get('selected_whitelist_blacklist_option') ?? 'whitelist',
        '#options' => [
            'whitelist' => $this->t('Allow'),
            'blacklist' => $this->t('Block'),
        ],
        '#disabled' => false,
    ];

    $form['ldap_group_restriction']['whitelist_textarea'] = [
        '#type' => 'textarea',
        '#title' => $this->t('<span style="color: green">Allowed LDAP Groups and OU</span>'),
        '#description' => $this->t('Only users belonging to the above entered LDAP groups or OU are allowed to log in to your Drupal site. Enter one per line such as <pre>cn=drupal_users,dc=example,dc=com<br>cn=admin_users,dc=example,dc=com</pre>'),
        '#states' => [
            'visible' => [
                ':input[name=whitelist_blacklist_radio_buttons]' => ['value' => 'whitelist'],
            ],
            'disabled' => [
                'input[name=group_restriction]'=> ['checked' => FALSE]
            ],
        ],
        '#rows' => 3,
    ];

    $form['ldap_group_restriction']['blacklist_textarea'] = [
        '#type' => 'textarea',
        '#title' => $this->t('<span style="color: red">Blocked LDAP Groups and OU</span>'),
        '#description' => $this->t('Above entered LDAP groups or OU users are <strong>NOT</strong> allowed to log in to your Drupal site. Enter one per line such as <pre>cn=restricted_users,dc=example,dc=com<br>cn=blocked_user,dc=example,dc=com</pre>'),
        '#states' => [
            'visible' => [
                ':input[name=whitelist_blacklist_radio_buttons]' => ['value' => 'blacklist']
            ],
            'disabled' => [
                'input[name=group_restriction]'=> ['checked' => FALSE]
            ]
        ],
        '#rows' => 3,
    ];

    $form['miniorange_save_advance_settings'] = [
        '#type' => 'submit',
        '#button_type' => 'primary',
        '#value' => t('Save Settings'),
    ];

    $form['markup_close'] = [
        '#type' => 'markup',
        '#markup' => '</div></div>'
    ];

    Utilities::addSupportButton( $form, $form_state);

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state) {

      $form_values = $form_state->getValues();
      $username_description = isset($form_values['username_description']) ? trim($form_values['username_description']) : "";
      $password_description = isset($form_values['password_description']) ? trim($form_values['password_description']) : "";

      $this->config_factory->set('username_description',$username_description)
                           ->set('password_description',$password_description)
                           ->save();

      $this->messenger->addStatus('Configuration saved successfully');

  }
  public function CustomFieldStorage() {

    $field_storage_configs = $this->entityTypeManager->getStorage('field_storage_config')->loadByProperties(['entity_type' => 'user']);

    $options = [
      'mail' => 'mail',
      'name' => 'name',
    ];

    foreach ($field_storage_configs as $field_storage) {
      $field_name = $field_storage->getName();

      if (!in_array($field_name, ['name', 'mail'])) {
        $options[$field_name] = $field_name;
      }
    }

    return $options;
  }

  private static $config_mapping = [

    "module_type" => [
      "ldap_module_type" => "miniorange_ldap_free_module",
    ],

    "ldap_configuration" => [

      // Contact LDAP server
      "ldap_protocol" => "miniorange_ldap_protocol",
      "ldap_server_address"  =>  "miniorange_ldap_server_address",
      "ldap_server_port_number" =>  "miniorange_ldap_server_port_number",

      // LDAP Binding
      "ldap_server_account_username" => "miniorange_ldap_server_account_username",
      "ldap_server_account_password" => "miniorange_ldap_server_account_password",

      // Search base and search filter
      "ldap_search_base" => "miniorange_ldap_search_base",
      "ldap_search_filter" => "miniorange_ldap_username_attribute_option",
      "ldap_email_attribute" => "miniorange_ldap_email_attribute",

      // Login settings
      "ldap_enable_ldap" => "miniorange_ldap_enable_ldap",

      // Extra key-value pair for the module need to recheck did they exist for this module or not
      "ldap_is_configured" => "miniorange_ldap_is_configured",
      "ldap_login_by" => "miniorange_ldap_login_by",
      "ldap_steps" => "miniorange_ldap_steps",
      "ldap_extension_enabled" =>  "miniorange_ldap_extension_enabled",
      "ldap_supports_anonymous_bind" =>  "supports_anonymous_bind",
      "ldap_contacted_server" => "miniorange_ldap_contacted_server",
      "ldap_config_status" =>  "miniorange_ldap_config_status",
    ],

    // key-value pair mapping for the NTLM & KERBEROS mapping
    "ntlm_kerberos" => [
      // This feature is not available for the free module
    ],

    "attribute_roleMapping" => [
      // This feature is not available for the free module
    ],

    // mapping the key-value pair for the advance_settings tab
    "advance_settings" => [

      // user login interface
      "ldap_username_description" => "username_description",
      "ldap_password_description" => "password_description",
    ],

    // mapping the key-value pair for the ldap_mappings tab
    "ldap_provisioning" => [
      // This feature is not available for the free module
    ],
  ];

  // Exportig the key-value pair values to the json file
  public static function exportConfigurationToJson() {

    $config_values = self::getConfigurationValues();
    self::outputJsonFile($config_values);

  }

  private static function getConfigurationValues() {
    // Fetching the config object once
    $config = \Drupal::config('ldap_auth.settings');
    $config_values = [];
    foreach (self::$config_mapping as $tab => $mappings) {
        $config_values[$tab] = [];
        foreach ($mappings as $config_key => $array_key) {

          // This condition is use to set the module type in json file
          if ($config_key === "ldap_module_type") {
            $config_values[$tab][$config_key] = $array_key;
          }
          // Combining the ldap protocol, server address, port number to make ldap_server use to import for premium and all-inclusive module
          elseif ($tab === "ldap_configuration" && $config_key === "ldap_protocol") {
            // Combine protocol, address, and port for ldap_server
            $protocol = $config->get($mappings["ldap_protocol"]);
            $address = $config->get($mappings["ldap_server_address"]);
            $port = $config->get($mappings["ldap_server_port_number"]);
            $config_values[$tab]["ldap_server"] = $protocol . $address . ":" . $port;

          }
          else
          {
            $value = $config->get($array_key);
            $config_values[$tab][$config_key] = $value;
          }
        }
    }
    return $config_values;
}

  private static function outputJsonFile($data) {
    // Convert data to JSON
    $json_data = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

    $filename =  'mo_free_module_configuration.json';

    // Create a response object
    $response = new Response($json_data);

    // Set headers for JSON file download
    $response->headers->set('Content-Type', 'application/json');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');
    LDAPLOGGERS::addLogger(MiniorangeLDAPConstants::EXPORT_CONFIGURATION,'Export Configuration Successfull');
    $response->send();
  }

  // Importing confirguration uploading only json file with key-pair values mapping
  public static function importConfiguration() {
    $request = \Drupal::request();
    $all_files = $request->files->get('files', []);

    // Validate file upload
    if (empty($all_files['import_Config_file']) || !$all_files['import_Config_file']->isValid()) {
      \Drupal::messenger()->addError(t('Import Configuration Failed'));
      LDAPLOGGERS::addLogger(MiniorangeLDAPConstants::EXPORT_CONFIGURATION, "No valid file uploaded. Please select a file first.");
      return;
    }
    $metadataFile = $all_files['import_Config_file'];
    $file_path = $metadataFile->getRealPath();
    if (!is_readable($file_path)) {
      \Drupal::messenger()->addError(t('Import Configuration Failed'));
      LDAPLOGGERS::addLogger(MiniorangeLDAPConstants::EXPORT_CONFIGURATION, "Unable to read the uploaded file");
      return;
    }
    $file_content = file_get_contents($file_path);
    if ($file_content === false) {
      \Drupal::messenger()->addError(t('Import Configuration Failed'));
      LDAPLOGGERS::addLogger(MiniorangeLDAPConstants::EXPORT_CONFIGURATION, "Error Reading the file content");
      return;
    }
    // Decode JSON and check for errors
    $config_values = json_decode($file_content, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
      \Drupal::messenger()->addError(t('Import Configuration Failed'));
      LDAPLOGGERS::addLogger(MiniorangeLDAPConstants::EXPORT_CONFIGURATION, "Import Configuration Failed");
      return;
    }
    // Update configuration
    self::updateConfiguration($config_values);
    \Drupal::messenger()->addStatus(t('Configuration imported successfully.'));
    LDAPLOGGERS::addLogger(MiniorangeLDAPConstants::EXPORT_CONFIGURATION, 'Configuration imported successfully.');
  }

  // This function helps us to update the config values
  private static function updateConfiguration(array $config_values) {
    // Get editable config object
    $config = \Drupal::configFactory()->getEditable('ldap_auth.settings');
    // Iterate through tabs and mappings
    foreach (self::$config_mapping as $tab => $mappings) {

      if (!empty($config_values[$tab])) {
        // Loop through key-value mappings
        foreach ($mappings as $config_key => $array_key) {
          // Set configuration value based on the mapping
          if (isset($config_values[$tab][$config_key])) {
            $config->set($array_key, $config_values[$tab][$config_key]);
          }
        }
      }
    }
    // Save the updated configuration
    $config->save();
  }
}
