<?php

namespace Drupal\leadsms_text\Form;

use Drupal\Component\Utility\Html;
use Drupal\Core\Condition\ConditionManager;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Render\Markup;
use GuzzleHttp\ClientInterface;
use Http\Client\Exception\RequestException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class LeadSmsSettingsForm.
 *
 * Provides a form for LEADsms settings configuration.
 */
class LeadSmsSettingsForm extends ConfigFormBase {

  /**
   * The condition plugin manager.
   *
   * @var \Drupal\Core\Condition\ConditionManager
   */
  protected $manager;

  /**
   * The context repository service.
   *
   * @var \Drupal\Core\Plugin\Context\ContextRepositoryInterface
   */
  protected $contextRepository;

  /**
   * The HTTP client for making requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The logger channel factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The host URL for CONNECTsms service.
   *
   * @var string
   */
  private $hostURL = 'https://connectsms.ca';

  /**
   * Constructs a new LeadSmsSettingsForm.
   *
   * @param \Drupal\Core\Condition\ConditionManager $condition_manager
   *   The condition plugin manager service.
   * @param \Drupal\Core\Plugin\Context\ContextRepositoryInterface $context_repository
   *   The context repository service.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client for making requests.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service for displaying messages.
   */
  public function __construct(ConditionManager $condition_manager,
    $context_repository,
    ClientInterface $http_client,
    LoggerChannelFactoryInterface $logger_factory,
    MessengerInterface $messenger
  ) {
    $this->manager = $condition_manager;
    $this->contextRepository = $context_repository;
    $this->httpClient = $http_client;
    $this->loggerFactory = $logger_factory;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.condition'),
      $container->get('context.repository'),
      $container->get('http_client'),
      $container->get('logger.factory'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'leadsms_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['leadsms_text.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Load the configuration.
    $config = $this->config('leadsms_text.settings');
    $status = $config->get('license.status');
    $saved_conditions = $config->get('visibility_conditions') ?? [];

    $form['container'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'actions-wrapper'],
    ];

    // The box contains some markup that we can change on a submit request.
    $form['container']['box'] = [
      '#type' => 'markup',
      '#markup' => '',
    ];

    // Initialize the vertical tabs.
    $form['#tree'] = TRUE;
    $form['#attributes']['novalidate'] = 'novalidate';
    $form['settings_tabs'] = [
      '#type' => 'vertical_tabs',
    ];

    // Add the tabs.
    $form['license_activation'] = $this->buildLicenseActivationTab($config);
    if ($status) {
      $form['visibility'] = $this->buildVisibilityInterface([], $form_state, $saved_conditions);
      $form['appearance'] = $this->buildAppearanceTab($config);
    }
    $form['instructions'] = $this->buildInstructionsTab();

    $form['actions_wrapper'] = [
      '#type' => 'container',
    ];

    if ($status) {
      // Add the submit button inside the wrapper.
      $form['actions_wrapper']['actions'] = [
        '#type' => 'button',
        '#value' => $this->t('Save Configuration'),
        '#ajax' => [
          'callback' => '::onAjaxSubmitForm',
          'wrapper' => 'actions-wrapper',
        ],
        '#attributes' => [
          'class' => ['button', 'button--primary'],
        ],
      ];
    }
    else {
      $form['save_config'] = [
        '#type' => 'submit',
        '#value' => $this->t('Activate License'),
      ];
    }

    $form['#attached']['library'][] = 'leadsms_text/leadsms-text';
    $form['#theme'] = 'leadsms_config_form';
    return $form;
  }

  /**
   * Builds the license activation tab.
   *
   * @param \Drupal\Core\Config\Config $config
   *   The configuration object.
   *
   * @return array
   *   The license activation form array.
   */
  private function buildLicenseActivationTab($config) {
    $status = $config->get('license.status');
    $form['license_activation'] = [
      '#type' => 'details',
      '#title' => $this->t('License and Activation'),
      '#group' => 'settings_tabs',
    ];

    // License and Activation Markup.
    $html_license_activation = <<<HTML
<div style="font-size: 14px">
    <h3 style="font-size: 15px"><strong>
    {$this->t('LEADsms integrates with your CONNECTsms account, allowing you to offer SMS/text help & support or generate leads from your Drupal site.')}
    </strong></h3>
    <p>
    {$this->t('To activate this plugin you will need to obtain your license key from your CONNECTsms account\'s settings accessed from the cog icon at the top right of the CONNECTsms application (look for "LEADsms License Key"). To login to your CONNECTsms account, <a href="@connectsms_url" target="_blank">click here</a>.', ['@connectsms_url' => 'https://connectsms.ca/'])}
    </p>
    <p>
    {$this->t('No account? To learn more about CONNECTsms, <a href="@localtext_url" target="_blank">click here</a>.', ['@localtext_url' => 'https://localtextmarketers.com/canadian-business-texting/'])}
    </p>
</div>
HTML;

    $form['license_activation']['header'] = [
      '#type' => 'markup',
      '#markup' => Markup::create($html_license_activation),
    ];

    // License Key Field.
    $form['license_activation']['license_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('License Key'),
      '#description' => $this->t('Please enter your license key.'),
      '#default_value' => $config->get('license.license_key'),
      '#required' => TRUE,
    ];

    if ($status) {
      $form['license_activation']['status'] = [
        '#type' => 'markup',
        '#markup' => '<div class="text-green">' . $this->t("Active.") . '</div>',
      ];
    }

    if ($status == 'invalid') {
      $form['license_activation']['status'] = [
        '#type' => 'markup',
        '#markup' => '<div class="text-red">' . $this->t("Invalid.") . '</div>',
      ];
    }

    return $form['license_activation'];
  }

  /**
   * Helper function for building the visibility UI form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param array $saved_conditions
   *   The saved visibility conditions from configuration.
   *
   * @return array
   *   The form array with the visibility UI added in.
   */
  private function buildVisibilityInterface(array $form, FormStateInterface $form_state, array $saved_conditions) {
    $allowed_conditions = [
      'entity_bundle:node',
      'request_path',
      'user_role',
    ];

    $form['visibility'] = [
      '#type' => 'details',
      '#title' => $this->t('Visibility Options'),
      '#group' => 'settings_tabs',
    ];

    $form['visibility']['header'] = [
      '#type' => 'markup',
      '#markup' => "Control when and where blocks appear with conditions based on content type, user roles, or paths. Without conditions, the block shows on every page.",
    ];

    // Create condition instances with saved values.
    foreach ($allowed_conditions as $condition_id) {
      $condition = $this->manager->createInstance($condition_id, $saved_conditions[$condition_id] ?? []);
      $form_state->set(['conditions', $condition_id], $condition);

      // Build the condition configuration form.
      $condition_form = $condition->buildConfigurationForm([], $form_state);
      $condition_form['#type'] = 'details';
      $condition_form['#title'] = $condition->getPluginDefinition()['label'];
      $form['visibility']['conditions'][$condition_id] = $condition_form;
    }

    // Disable negation for specific conditions.
    $disable_negation = [
      'entity_bundle:node',
      'user_role',
    ];

    foreach ($disable_negation as $condition) {
      if (isset($form['visibility']['conditions'][$condition])) {
        $form['visibility']['conditions'][$condition]['negate']['#type'] = 'value';
        $form['visibility']['conditions'][$condition]['negate']['#value'] = $form['visibility']['conditions'][$condition]['negate']['#default_value'];
      }
    }

    return $form['visibility'];
  }

  /**
   * Builds the appearance tab for the widget configuration.
   *
   * @param \Drupal\Core\Config\Config $config
   *   The configuration object.
   *
   * @return array
   *   The appearance configuration form array.
   */
  private function buildAppearanceTab($config) {
    $form['appearance'] = [
      '#type' => 'details',
      '#title' => $this->t('Appearance'),
      '#group' => 'settings_tabs',
      '#open' => TRUE,
    ];

    $form['appearance']['description'] = [
      '#type' => 'markup',
      '#markup' => $this->t('Customize the look and feel of your text widget.'),
    ];

    $form['appearance']['header_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Header Text'),
      '#default_value' => $config->get('header_text'),
      '#ajax' => [
        'callback' => '::updateWidget',
        'event' => 'change',
        'progress' => [
          'type' => 'none',
        ],
      ],
    ];

    // Add a fieldset for width and height.
    $form['appearance']['dimensions'] = [
      '#type' => 'details',
      '#title' => $this->t('Widget Dimensions'),
      '#open' => FALSE,
    ];

    // Add width number field.
    $form['appearance']['dimensions']['width'] = [
      '#type' => 'number',
      '#title' => $this->t('Width'),
      '#default_value' => $config->get('widget_width'),
      '#size' => 5,
      '#min' => 100,
      '#max' => 1000,
      '#wrapper_attributes' => [
        'style' => 'display: inline-block; width: auto; margin-right: 10px;',
      ],
      '#ajax' => [
        'callback' => '::updateWidget',
        'event' => 'change',
        'progress' => [
          'type' => 'none',
        ],
      ],
    ];

    // Add height number field.
    $form['appearance']['dimensions']['height'] = [
      '#type' => 'number',
      '#title' => $this->t('Height'),
      '#default_value' => $config->get('widget_height'),
      '#size' => 5,
      '#min' => 100,
      '#max' => 1000,
      '#wrapper_attributes' => [
        'style' => 'display: inline-block; width: auto; margin-right: 10px;',
      ],
      '#ajax' => [
        'callback' => '::updateWidget',
        'event' => 'change',
        'progress' => [
          'type' => 'none',
        ],
      ],
    ];

    $form['appearance']['widget_form_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Widget Form Settings'),
    ];

    // Header.
    $form['appearance']['widget_form_settings']['header'] = [
      '#type' => 'fieldset',
      '#title' => 'Header',
    ];

    // Header Background Color.
    $form['appearance']['widget_form_settings']['header']['header_bg'] = [
      '#type' => 'color',
      '#title' => $this->t('Background Color'),
      '#default_value' => $config->get('header_bg'),
      '#wrapper_attributes' => [
        'style' => 'display: inline-block; width: auto; margin-right: 20px;',
      ],
      '#ajax' => [
        'callback' => '::updateWidget',
        'event' => 'change',
        'progress' => [
          'type' => 'none',
        ],
      ],
    ];

    // Header Text Color.
    $form['appearance']['widget_form_settings']['header']['header_text_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Text Color'),
      '#default_value' => $config->get('header_text_color'),
      '#wrapper_attributes' => [
        'style' => 'display: inline-block; width: auto; margin-right: 20px;',
      ],
      '#ajax' => [
        'callback' => '::updateWidget',
        'event' => 'change',
        'progress' => [
          'type' => 'none',
        ],
      ],
    ];

    $form['appearance']['widget_form_settings']['buttons'] = [
      '#type' => 'fieldset',
      '#title' => 'Buttons',
    ];

    // Submit Button Color.
    $form['appearance']['widget_form_settings']['buttons']['submit_bg'] = [
      '#type' => 'color',
      '#title' => $this->t('Submit Button Color'),
      '#default_value' => $config->get('submit_bg'),
      '#wrapper_attributes' => [
        'style' => 'display: inline-block; width: auto; margin-right: 20px;',
      ],
      '#ajax' => [
        'callback' => '::updateWidget',
        'event' => 'change',
        'progress' => [
          'type' => 'none',
        ],
      ],
    ];

    $form['appearance']['widget_form_settings']['buttons']['submit_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Submit Button Text'),
      '#maxlength' => 15,
      '#size' => 15,
      '#default_value' => $config->get('submit_text'),
      '#wrapper_attributes' => [
        'style' => 'display: inline-block; width: auto; margin-right: 20px; vertical-align: top',
      ],
      '#ajax' => [
        'callback' => '::updateWidget',
        'event' => 'change',
        'progress' => [
          'type' => 'none',
        ],
      ],
    ];

    $form['appearance']['widget_form_settings']['buttons']['launcher_bg'] = [
      '#type' => 'color',
      '#title' => $this->t('Text Us Button Color'),
      '#default_value' => $config->get('launcher_bg'),
      '#wrapper_attributes' => [
        'style' => 'display: inline-block; width: auto; margin-right: 20px;',
      ],
      '#ajax' => [
        'callback' => '::updateWidget',
        'event' => 'change',
        'progress' => [
          'type' => 'none',
        ],
      ],
    ];

    $form['appearance']['widget_form_settings']['buttons']['launcher_text_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Text Us Button Text Color'),
      '#default_value' => $config->get('launcher_text_color'),
      '#wrapper_attributes' => [
        'style' => 'display: inline-block; width: auto; margin-right: 20px;',
      ],
      '#ajax' => [
        'callback' => '::updateWidget',
        'event' => 'change',
        'progress' => [
          'type' => 'none',
        ],
      ],
    ];

    $form['appearance']['widget_form_settings']['body'] = [
      '#type' => 'fieldset',
      '#title' => 'Body',
    ];

    // Body Background Color.
    $form['appearance']['widget_form_settings']['body']['body_bg'] = [
      '#type' => 'color',
      '#title' => $this->t('Background Color'),
      '#default_value' => $config->get('body_bg'),
      '#wrapper_attributes' => [
        'style' => 'display: inline-block; width: auto; margin-right: 20px;',
      ],
      '#ajax' => [
        'callback' => '::updateWidget',
        'event' => 'change',
        'progress' => [
          'type' => 'none',
        ],
      ],
    ];

    $form['appearance']['popover_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Popover Settings'),
    ];

    // Alert Box with Information.
    $form['appearance']['popover_settings']['popover_info'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--warning">' . $this->t("Pops open a small window to draw the visitor's attention to the contact form option. We recommend using a minimum of a 10-15 second delay for your Popover.") . '</div>',
    ];

    // Enable Popover Toggle.
    $form['appearance']['popover_settings']['popover'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Popover'),
      '#default_value' => $config->get('popover'),
    ];

    $form['appearance']['popover_settings']['popover_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Content'),
      '#default_value' => $config->get('popover_text'),
      '#maxlength' => 150,
      '#ajax' => [
        'callback' => '::updateWidget',
        'event' => 'change',
        'progress' => [
          'type' => 'none',
        ],
      ],
    ];

    $form['appearance']['popover_settings']['popover_delay'] = [
      '#type' => 'number',
      '#title' => $this->t('Delay (in seconds)'),
      '#default_value' => $config->get('popover_delay'),
      '#min' => 1,
      '#size' => 4,
    ];

    return $form['appearance'];
  }

  /**
   * Builds the instructions tab for the configuration form.
   *
   * @return array
   *   The instructions tab form array.
   */
  private function buildInstructionsTab() {
    $form['instructions'] = [
      '#type' => 'details',
      '#title' => $this->t('Instructions'),
      '#group' => 'settings_tabs',
    ];

    $html_instructions = <<<HTML
    <div style="font-size: 14px">
        <div>
            <h3 style="font-size: 15px"><strong>
            {$this->t('To enable the LEADsms text widget on your Drupal website...')}
            </strong></h3>
        </div>
        <div>
            <ul>
                <li><span>1)</span> <span>
                {$this->t('Obtain your LEADsms License Key from your CONNECTsms account\'s settings page and save it under "License and Activation" here in Drupal.')}
                </span></li>
                <li><span>2)</span> <span>
                {$this->t('Select your visibility options.')}
                </span></li>
                <li><span>3)</span> <span>
                {$this->t('Customize the appearance of your text widget.')}
                </span></li>
                <li><span>4)</span> <span>
                {$this->t('That\'s it. Now just wait for the messages to arrive in CONNECTsms and respond as usual (messages from new contacts will appear in the External Inbox, while messaging from existing ones will be routed directly to the correct User/Agent).')}
                </span></li>
                <li><span>5)</span> <span>
                {$this->t('TIP: To ensure your contacts are responded to quickly, be sure to enable all "new message notifications" in CONNECTsms\' settings.')}
                </span></li>
            </ul>
        </div>
        <div>
            <p>
            {$this->t('For help and support, please <a href="@support_url" target="_blank">contact us</a>.', ['@support_url' => 'https://localtextmarketers.com/contact-us/'])}
            </p>
        </div>
    </div>
HTML;

    // Add the markup to the instructions tab.
    $form['instructions']['markup'] = [
      '#type' => 'markup',
      '#markup' => Markup::create($html_instructions),
    ];

    return $form['instructions'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $form_state->setValue(['license_status'], FALSE);

    // Sanitize input using Html::escape().
    $license_key = Html::escape($form_state->getValue([
      'license_activation',
      'license_key',
    ]));

    // Define the endpoint URL.
    $endpoint = $this->hostURL . "/_LEADsms_LK_validate.php";

    // Define the data to be sent with the request.
    $post_data = [
      'licensekey' => $license_key,
      'leadsmsaccess' => '3Txtea6BDo6naRjDwm3pUM42z8Udr7EC',
    ];

    try {
      // Make the POST request using Guzzle.
      $response = $this->httpClient->post($endpoint, [
        'form_params' => $post_data,
      ]);

      // Check the response status code.
      if ($response->getStatusCode() == 200) {
        // Decode the response if it's JSON.
        $response_body = json_decode($response->getBody(), TRUE);

        // Process the response (e.g., validate license).
        if (!empty($response_body['valid']) && $response_body['valid'] == TRUE) {
          $form_state->setValue(['license_status'], 'valid');
          $form_state->setValue(['hide_powered_by'], (int) $response_body['hide_powered_by']);
        }
        else {
          $form_state->setErrorByName('license_key', $this->t('The license is invalid.'));
          $form_state->setValue(['license_status'], 'invalid');
        }
      }
    }
    catch (RequestException $e) {
      // Handle the exception and log the error using the injected logger.
      $this->loggerFactory->get('leadsms_text')
        ->error('License validation failed: @message', ['@message' => $e->getMessage()]);

      // Display an error message to the user using the injected messenger.
      $this->messenger->addError($this->t('There was an error validating the license key. Please try again.'));
    }
  }

  /**
   * Saves license key.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $license['license_key'] = $form_state->getValue([
      'license_activation',
      'license_key',
    ]);
    $license['status'] = $form_state->getValue('license_status');
    $hide_powered_by = $form_state->getValue('hide_powered_by');
    $this->config('leadsms_text.settings')->set('license', $license)->save();
    $this->config('leadsms_text.settings')->set('hide_powered_by', $hide_powered_by)->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * Saves widget config values.
   *
   * {@inheritdoc}
   */
  public function onAjaxSubmitForm(array &$form, FormStateInterface $form_state) {
    $errors = $form_state->getErrors();
    if (isset($errors['license_key'])) {
      return $form['container'];
    }

    $visibility_conditions = [];

    $conditions = $form_state->getValue([
      'visibility',
      'conditions',
    ]);

    if (is_array($conditions)) {
      foreach ($conditions as $condition_id => $values) {
        // Get the condition instance from the form state.
        $condition = $form_state->get(['conditions', $condition_id]);

        if ($condition) {
          $subform_state = SubformState::createForSubform($form['visibility']['conditions'][$condition_id], $form, $form_state);
          $condition->submitConfigurationForm($form['visibility']['conditions'][$condition_id], $subform_state);

          // Store the condition configuration after applying submitted values.
          $condition_configuration = $condition->getConfiguration();

          // Check if the condition has a 'negate' field and apply it.
          if (isset($values['negate'])) {
            $condition_configuration['negate'] = (bool) $values['negate'];
          }
          else {
            $condition_configuration['negate'] = FALSE;
          }

          // Save the condition configuration, including negation.
          $visibility_conditions[$condition_id] = $condition_configuration;
        }
      }
    }

    $license['license_key'] = $form_state->getValue([
      'license_activation',
      'license_key',
    ]);
    $license['status'] = $form_state->getValue('license_status');
    $hide_powered_by = $form_state->getValue('hide_powered_by');
    $widget_config = $this->getWidgetConfig($form_state);

    // Set the merged data.
    $this->config('leadsms_text.settings')->setData($widget_config)
      ->set('license', $license)
      ->set('visibility_conditions', $visibility_conditions)
      ->set('hide_powered_by', $hide_powered_by)
      ->save();

    $this->messenger->addStatus($this->t('The configuration options have been saved.'));
    return $form['container'];
  }

  /**
   * Widget preview.
   */
  public function updateWidget(array &$form, FormStateInterface $form_state) {
    $widget_config = $this->getWidgetConfig($form_state);
    $form['#attached']['drupalSettings']['leadsms'] = $widget_config;
    return $form;
  }

  /**
   * Prepares widget config data.
   */
  public function getWidgetConfig(FormStateInterface $form_state) {
    return [
      'header_text' => $form_state->getValue(['appearance', 'header_text']),
      'widget_width' => $form_state->getValue([
        'appearance',
        'dimensions',
        'width',
      ]),
      'widget_height' => $form_state->getValue([
        'appearance',
        'dimensions',
        'height',
      ]),
      'header_bg' => $form_state->getValue([
        'appearance',
        'widget_form_settings',
        'header',
        'header_bg',
      ]),
      'header_text_color' => $form_state->getValue([
        'appearance',
        'widget_form_settings',
        'header',
        'header_text_color',
      ]),
      'submit_bg' => $form_state->getValue([
        'appearance',
        'widget_form_settings',
        'buttons',
        'submit_bg',
      ]),
      'submit_text' => $form_state->getValue([
        'appearance',
        'widget_form_settings',
        'buttons',
        'submit_text',
      ]),
      'body_bg' => $form_state->getValue([
        'appearance',
        'widget_form_settings',
        'body',
        'body_bg',
      ]),
      'launcher_bg' => $form_state->getValue([
        'appearance',
        'widget_form_settings',
        'buttons',
        'launcher_bg',
      ]),
      'launcher_text_color' => $form_state->getValue([
        'appearance',
        'widget_form_settings',
        'buttons',
        'launcher_text_color',
      ]),
      'popover' => $form_state->getValue([
        'appearance',
        'popover_settings',
        'popover',
      ]),
      'popover_text' => $form_state->getValue([
        'appearance',
        'popover_settings',
        'popover_text',
      ]),
      'popover_delay' => $form_state->getValue([
        'appearance',
        'popover_settings',
        'popover_delay',
      ]),
    ];
  }

}
