/**
 * @file
 * JavaScript for Leaflet Dynamic Attachment.
 *
 * Updates the dynamic attachment table when the Leaflet map viewport changes.
 */

(function ($, Drupal, drupalSettings, once) {
  'use strict';

  /**
   * Behavior for Leaflet Dynamic Attachment.
   */
  Drupal.behaviors.leafletDynamicAttachment = {
    attach: function (context, settings) {
      // Check for our settings.
      if (!settings.leafletDynamicAttachment) {
        return;
      }

      var config = settings.leafletDynamicAttachment;

      // Find dynamic attachment containers.
      once('leaflet-dynamic-attachment', '.leaflet-dynamic-attachment', context).forEach(function (container) {
        // Wait for Leaflet maps to initialize.
        Drupal.leafletDynamicAttachment.waitForMap(config, $(container));
      });
    }
  };

  /**
   * Leaflet Dynamic Attachment namespace.
   */
  Drupal.leafletDynamicAttachment = {
    map: null,
    leafletInstance: null,
    initialized: false,

    /**
     * Wait for Leaflet map to be available.
     */
    waitForMap: function (config, container) {
      var self = this;
      var maxAttempts = 20;
      var attempts = 0;

      var checkMap = function () {
        attempts++;

        if (window.Drupal && window.Drupal.Leaflet) {
          var mapInstances = Object.keys(window.Drupal.Leaflet);
          if (mapInstances.length > 0) {
            var mapId = mapInstances[0];
            var leafletInstance = window.Drupal.Leaflet[mapId];

            if (leafletInstance && leafletInstance.lMap) {
              self.init(config, container, leafletInstance);
              return;
            }
          }
        }

        if (attempts < maxAttempts) {
          setTimeout(checkMap, 500);
        }
        else {
          console.warn('Leaflet Dynamic Attachment: Could not find Leaflet map after', maxAttempts, 'attempts');
        }
      };

      checkMap();
    },

    /**
     * Initialize the dynamic attachment.
     */
    init: function (config, container, leafletInstance) {
      var self = this;

      if (this.initialized) {
        return;
      }
      this.initialized = true;
      this.leafletInstance = leafletInstance;
      this.map = leafletInstance.lMap;

      console.log('Leaflet Dynamic Attachment: Initialized');

      var debounceDelay = config.debounceDelay || 300;
      var updateTimer = null;

      // Debounced update function.
      var debouncedUpdate = function () {
        clearTimeout(updateTimer);
        updateTimer = setTimeout(function () {
          self.updateAttachment(config, container);
        }, debounceDelay);
      };

      // Listen for map events.
      if (config.updateOnZoom !== false) {
        this.map.on('zoomend', debouncedUpdate);
      }
      if (config.updateOnPan !== false) {
        this.map.on('moveend', debouncedUpdate);
      }

      // Perform initial update.
      this.updateAttachment(config, container);
    },

    /**
     * Get entity IDs from markers visible in the current map bounds.
     *
     * @returns {Array}
     *   Array of entity IDs currently visible on the map.
     */
    getVisibleEntityIds: function () {
      var entityIds = [];
      var bounds = this.map.getBounds();

      if (this.leafletInstance && this.leafletInstance.features) {
        // Iterate through all features and check if they're within bounds.
        for (var key in this.leafletInstance.features) {
          if (this.leafletInstance.features.hasOwnProperty(key)) {
            var feature = this.leafletInstance.features[key];
            if (feature && feature.entity_id && feature.lat && feature.lon) {
              // Check if this feature is within the current map bounds.
              var latLng = L.latLng(feature.lat, feature.lon);
              if (bounds.contains(latLng)) {
                // Avoid duplicates (from multi-value geofields).
                if (entityIds.indexOf(feature.entity_id) === -1) {
                  entityIds.push(feature.entity_id);
                }
              }
            }
          }
        }
      }

      return entityIds;
    },

    /**
     * Update the dynamic attachment via AJAX.
     */
    updateAttachment: function (config, container) {
      var self = this;

      // Get entity IDs from visible markers within current bounds.
      var entityIds = this.getVisibleEntityIds();

      console.log('Leaflet Dynamic Attachment: Updating with', entityIds.length, 'visible entity IDs');

      // Show loading indicator.
      container.addClass('is-loading');

      $.ajax({
        url: config.ajaxUrl,
        type: 'POST',
        data: {
          view_id: config.viewId,
          display_id: config.displayId,
          entity_ids: entityIds
        },
        success: function (response) {
          console.log('Leaflet Dynamic Attachment: Received response, count:', response.count);

          if (response.html) {
            // Update the container content but preserve the wrapper.
            container.html(response.html);

            // Add/update count display.
            var countDisplay = container.find('.leaflet-dynamic-count');
            if (countDisplay.length === 0) {
              container.prepend('<div class="leaflet-dynamic-count">' + response.count + ' items in view</div>');
            }
            else {
              countDisplay.text(response.count + ' items in view');
            }

            // Re-attach Drupal behaviors to new content.
            Drupal.attachBehaviors(container[0], drupalSettings);
          }
          container.removeClass('is-loading');
        },
        error: function (xhr, status, error) {
          console.error('Leaflet Dynamic Attachment: AJAX error', error);
          container.removeClass('is-loading');
        }
      });
    }
  };

})(jQuery, Drupal, drupalSettings, once);
