<?php

namespace Drupal\leaflet_dynamic_table\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Render\RendererInterface;
use Drupal\views\Views;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for Leaflet Dynamic Table AJAX updates.
 */
class LeafletDynamicTableController extends ControllerBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a LeafletDynamicTableController object.
   *
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(RendererInterface $renderer) {
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('renderer')
    );
  }

  /**
   * Handles AJAX update requests for the dynamic attachment.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing the rendered view HTML and item count.
   */
  public function update(Request $request) {
    // Get parameters from POST.
    $view_id = $request->request->get('view_id');
    $display_id = $request->request->get('display_id');
    $entity_ids = $request->request->all()['entity_ids'] ?? [];

    // Validate and sanitize view_id - must be alphanumeric with underscores.
    if (empty($view_id) || !preg_match('/^[a-z0-9_]+$/i', $view_id)) {
      return new JsonResponse([
        'error' => 'Invalid view ID.',
      ], 400);
    }

    // Validate and sanitize display_id - must be alphanumeric with underscores.
    if (empty($display_id) || !preg_match('/^[a-z0-9_]+$/i', $display_id)) {
      return new JsonResponse([
        'error' => 'Invalid display ID.',
      ], 400);
    }

    // Load the view.
    $view = Views::getView($view_id);
    if (!$view) {
      return new JsonResponse([
        'error' => 'View not found.',
      ], 404);
    }

    // Check if the user has access to the view.
    if (!$view->access($display_id)) {
      return new JsonResponse([
        'error' => 'Access denied.',
      ], 403);
    }

    // Store entity IDs for hook_views_query_alter.
    // We explicitly set to empty array when no markers visible, to distinguish
    // from NULL (not our AJAX request). This ensures 0 results are returned.
    // Sanitize entity_ids to integers only (prevents SQL injection).
    $entity_ids_data = &drupal_static('leaflet_dynamic_table_entity_ids', NULL);
    $entity_ids_data = !empty($entity_ids)
      ? array_map('intval', array_filter($entity_ids, 'is_numeric'))
      : [];

    // Set display and execute.
    $view->setDisplay($display_id);

    // Disable the pager for this AJAX request - show all filtered results.
    $view->setItemsPerPage(0);

    $view->preExecute();
    $view->execute();

    // Render the view.
    $render_array = $view->render();
    $html = $this->renderer->renderRoot($render_array);

    return new JsonResponse([
      'html' => (string) $html,
      'count' => count($view->result),
    ]);
  }

}
