<?php

namespace Drupal\leaflet_dynamic_table\Plugin\views\display;

use Drupal\views\Plugin\views\display\Attachment;
use Drupal\views\ViewExecutable;
use Drupal\Core\Form\FormStateInterface;

/**
 * A display plugin that attaches to a Leaflet map and updates dynamically.
 *
 * This attachment display is specifically designed to work with Leaflet map
 * displays. It automatically syncs with the map viewport - showing only the
 * items currently visible on the map. The pager and filter inheritance options
 * are removed since this display always follows what the map shows.
 *
 * @ingroup views_display_plugins
 *
 * @ViewsDisplay(
 *   id = "leaflet_dynamic_attachment",
 *   title = @Translation("Leaflet Dynamic Attachment"),
 *   help = @Translation("An attachment display that updates dynamically based on Leaflet map viewport. Only attaches to Leaflet map displays."),
 *   theme = "views_view",
 *   contextual_links_locations = {""}
 * )
 */
class LeafletDynamicAttachment extends Attachment {

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();

    // Leaflet-specific options.
    $options['leaflet_map_display'] = ['default' => ''];
    $options['update_on_zoom'] = ['default' => TRUE];
    $options['update_on_pan'] = ['default' => TRUE];
    $options['debounce_delay'] = ['default' => 300];

    // Force disable pager - we follow the map.
    $options['pager']['contains']['type']['default'] = 'none';

    // Disable filter/argument inheritance - we follow the map.
    $options['inherit_arguments']['default'] = FALSE;
    $options['inherit_exposed_filters']['default'] = FALSE;

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function optionsSummary(&$categories, &$options) {
    parent::optionsSummary($categories, $options);

    // Add our Leaflet Dynamic category.
    $categories['leaflet_dynamic'] = [
      'title' => $this->t('Leaflet Dynamic'),
      'column' => 'second',
      'build' => [
        '#weight' => -10,
      ],
    ];

    $map_display = $this->getOption('leaflet_map_display');
    $options['leaflet_map_display'] = [
      'category' => 'leaflet_dynamic',
      'title' => $this->t('Map display'),
      'value' => $map_display ?: $this->t('Not set'),
    ];

    $options['leaflet_dynamic_settings'] = [
      'category' => 'leaflet_dynamic',
      'title' => $this->t('Dynamic settings'),
      'value' => $this->t('Debounce: @ms ms', ['@ms' => $this->getOption('debounce_delay')]),
    ];

    // Remove pager option from summary - not applicable.
    unset($options['pager']);

    // Remove all pager-related options - we follow the map.
    unset($options['use_more']);
    unset($options['use_more_always']);
    unset($options['use_more_text']);
    unset($options['link_display']);
    unset($options['inherit_pager']);
    unset($options['render_pager']);

    // Remove inheritance options - we always follow the map.
    unset($options['inherit_arguments']);
    unset($options['inherit_exposed_filters']);

    // Modify the "Attach to" description to clarify it's Leaflet-only.
    if (isset($options['displays'])) {
      $options['displays']['title'] = $this->t('Attach to Leaflet map');
    }

    // Remove the entire Pager category since it's not used.
    unset($categories['pager']);

    // Remove Advanced options that don't apply to this display type.
    // We handle AJAX ourselves, so no need for Views AJAX.
    unset($options['use_ajax']);
    // Contextual links not useful for dynamic attachment.
    unset($options['show_admin_links']);
    // No aggregation needed - we filter by entity IDs from the map.
    unset($options['group_by']);
    // Query settings not needed for this simple filtering.
    unset($options['query']);
    // Caching bypassed by our AJAX updates.
    unset($options['cache']);
    // Exposed form not needed - filters come from map viewport.
    unset($options['exposed_form']);

    // Remove handler sections from Advanced that aren't needed.
    // Relationships not needed - we filter by entity IDs directly.
    unset($options['relationships']);
    // Contextual filters not needed - we follow the map, not URL args.
    unset($options['arguments']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    // Attach admin CSS to hide unnecessary sections.
    $form['#attached']['library'][] = 'leaflet_dynamic_table/admin';

    $section = $form_state->get('section');

    switch ($section) {
      case 'leaflet_map_display':
        $form['#title'] .= $this->t('Leaflet Map Display');
        $displays = $this->getLeafletMapDisplays();
        $form['leaflet_map_display'] = [
          '#type' => 'select',
          '#title' => $this->t('Leaflet Map Display'),
          '#description' => $this->t('Select the Leaflet map display this attachment should respond to.'),
          '#options' => $displays,
          '#default_value' => $this->getOption('leaflet_map_display'),
          '#required' => TRUE,
        ];
        break;

      case 'leaflet_dynamic_settings':
        $form['#title'] .= $this->t('Dynamic Update Settings');
        $form['update_on_zoom'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Update on zoom'),
          '#default_value' => $this->getOption('update_on_zoom'),
        ];
        $form['update_on_pan'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Update on pan'),
          '#default_value' => $this->getOption('update_on_pan'),
        ];
        $form['debounce_delay'] = [
          '#type' => 'number',
          '#title' => $this->t('Debounce delay (ms)'),
          '#description' => $this->t('Delay in milliseconds before updating after map movement stops.'),
          '#default_value' => $this->getOption('debounce_delay'),
          '#min' => 0,
          '#max' => 2000,
        ];
        break;

      case 'displays':
        // Override the displays form to only show Leaflet map displays.
        $form['#title'] .= $this->t('Attach to Leaflet Map Display');
        $displays = $this->getLeafletMapDisplays();

        if (empty($displays)) {
          $form['displays'] = [
            '#markup' => '<p>' . $this->t('No Leaflet map displays found in this view. Please create a Page or Block display with Leaflet map format first.') . '</p>',
          ];
        }
        else {
          $form['displays'] = [
            '#title' => $this->t('Attach to Leaflet map displays'),
            '#description' => $this->t('Select which Leaflet map display(s) this attachment should be attached to. This attachment will only work with displays using the Leaflet map format.'),
            '#type' => 'checkboxes',
            '#options' => $displays,
            '#default_value' => $this->getOption('displays'),
          ];
        }
        break;

      case 'pager':
        // Override pager form to show explanation.
        $form = [];
        $form['#title'] = $this->t('Pager');
        $form['pager_info'] = [
          '#markup' => '<p>' . $this->t("The pager is disabled for Leaflet Dynamic Attachment. This display automatically shows all items currently visible on the map, following the map's own pager settings.") . '</p>',
        ];
        break;

      case 'inherit_arguments':
      case 'inherit_exposed_filters':
        // Override to show explanation.
        $form = [];
        $form['#title'] = $this->t('Filter Inheritance');
        $form['info'] = [
          '#markup' => '<p>' . $this->t('Filter inheritance is disabled for Leaflet Dynamic Attachment. This display automatically follows the map viewport and shows only items visible on the map.') . '</p>',
        ];
        break;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state) {
    $section = $form_state->get('section');

    // Skip parent submission for disabled sections.
    if (in_array($section, ['pager', 'inherit_arguments', 'inherit_exposed_filters'])) {
      return;
    }

    parent::submitOptionsForm($form, $form_state);

    switch ($section) {
      case 'leaflet_map_display':
        $this->setOption('leaflet_map_display', $form_state->getValue('leaflet_map_display'));
        break;

      case 'leaflet_dynamic_settings':
        $this->setOption('update_on_zoom', $form_state->getValue('update_on_zoom'));
        $this->setOption('update_on_pan', $form_state->getValue('update_on_pan'));
        $this->setOption('debounce_delay', $form_state->getValue('debounce_delay'));
        break;
    }
  }

  /**
   * Gets available Leaflet map displays from this view.
   *
   * @return array
   *   An array of display titles keyed by display ID.
   */
  protected function getLeafletMapDisplays() {
    $displays = [];
    $view_displays = $this->view->storage->get('display');

    foreach ($view_displays as $display_id => $display) {
      // Skip default display.
      if ($display_id === 'default') {
        continue;
      }

      // Check if this display uses Leaflet map style.
      $is_leaflet = FALSE;

      // Check in display_options directly.
      if (isset($display['display_options']['style']['type']) &&
          $display['display_options']['style']['type'] === 'leaflet_map') {
        $is_leaflet = TRUE;
      }

      // Also check via display handler.
      if (!$is_leaflet) {
        $display_handler = $this->view->displayHandlers->get($display_id);
        if ($display_handler) {
          $style = $display_handler->getOption('style');
          if (isset($style['type']) && $style['type'] === 'leaflet_map') {
            $is_leaflet = TRUE;
          }
        }
      }

      if ($is_leaflet) {
        $displays[$display_id] = $display['display_title'] ?? $display_id;
      }
    }

    return $displays;
  }

  /**
   * {@inheritdoc}
   */
  public function attachTo(ViewExecutable $view, $display_id, array &$build) {
    // Check if we should attach to this display.
    $displays = $this->getOption('displays');
    if (empty($displays[$display_id])) {
      return;
    }

    // Verify the target display is a Leaflet map.
    $leaflet_displays = $this->getLeafletMapDisplays();
    if (!isset($leaflet_displays[$display_id])) {
      return;
    }

    // Check access.
    if (!$this->access()) {
      return;
    }

    // Set up the view for this display - no argument inheritance.
    $view->setArguments([]);
    $view->setDisplay($this->display['id']);

    // Build the attachment render array.
    $attachment_build = $view->buildRenderable($this->display['id'], []);

    // Wrap in a container with our dynamic class and data attributes.
    $wrapper = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['leaflet-dynamic-attachment'],
        'data-view-id' => $this->view->storage->id(),
        'data-display-id' => $this->display['id'],
      ],
      'content' => $attachment_build,
    ];

    // Add our JavaScript library and settings.
    $wrapper['#attached']['library'][] = 'leaflet_dynamic_table/dynamic-attachment';
    $wrapper['#attached']['drupalSettings']['leafletDynamicAttachment'] = [
      'viewId' => $this->view->storage->id(),
      'displayId' => $this->display['id'],
      'updateOnZoom' => (bool) $this->getOption('update_on_zoom'),
      'updateOnPan' => (bool) $this->getOption('update_on_pan'),
      'debounceDelay' => (int) $this->getOption('debounce_delay'),
      'ajaxUrl' => '/leaflet-dynamic-table/update',
    ];

    // Add to the view's attachment area using standard mechanism.
    $attachment_position = $this->getOption('attachment_position');
    switch ($attachment_position) {
      case 'before':
        $this->view->attachment_before[] = $wrapper;
        break;

      case 'after':
        $this->view->attachment_after[] = $wrapper;
        break;

      case 'both':
        $this->view->attachment_before[] = $wrapper;
        $this->view->attachment_after[] = $wrapper;
        break;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function usesPager() {
    // Disable pager - we follow the map.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function usesAttachments() {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function displaysAttachments() {
    return TRUE;
  }

}
