<?php

namespace Drupal\library_renderer\EventSubscriber;

use Drupal\Component\Utility\Html;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Render\HtmlResponse;
use Drupal\Core\Render\Markup;
use Drupal\library_renderer\LibraryRenderer;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Injects libraries based on html tag.
 */
class LibraryRendererSubscriber implements EventSubscriberInterface {

  /**
   * The component library parser.
   *
   * @var \Drupal\library_renderer\LibraryRenderer
   */
  private LibraryRenderer $libraryRenderer;

  /**
   * Constructs a new event subscriber.
   */
  public function __construct(LibraryRenderer $libraryRenderer) {
    $this->libraryRenderer = $libraryRenderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::RESPONSE => ['onKernelResponse', 30],
    ];
  }

  /**
   * Kernel response event handler.
   */
  public function onKernelResponse(ResponseEvent $event): void {
    $response = $event->getResponse();
    if ($response instanceof HtmlResponse) {
      $this->htmlResponse($response);
    }
    if ($response instanceof AjaxResponse) {
      $this->ajaxResponse($response);
    }
  }

  /**
   * Check response for html containing font awesome icons.
   */
  public function htmlResponse(HtmlResponse $response): void {
    $content = $response->getContent();
    $this->attachHtmlTagLibraries($content, $response);
  }

  /**
   * Check response for ajax commands containing font awesome icons.
   */
  public function ajaxResponse(AjaxResponse $response): void {
    $commands = $response->getCommands();
    // Iterate over the commands and check for Drupal\core\render\Markup.
    foreach ($commands as $command) {
      $command_data = (array) $command;
      foreach ($command_data as $content) {
        if ($content instanceof Markup) {
          $this->attachHtmlTagLibraries($content, $response);
          break;
        }
      }
    }
  }

  /**
   * Check if content contains font awesome icons.
   */
  public function attachHtmlTagLibraries($content, HtmlResponse|AjaxResponse $response): HtmlResponse|AjaxResponse {
    if (!empty($content)) {
      $hooked_libraries = $this->libraryRenderer->getHookedLibraries();
      if (!empty($hooked_libraries['html_tags'])) {
        $attachments = $response->getAttachments();
        $html = Html::load($content);
        foreach ($hooked_libraries['html_tags'] as $tag => $libraries) {
          $tag_elements = $html->getElementsByTagName($tag);
          if ($tag_elements->length !== 0) {
            foreach ($libraries as $library) {
              if (!empty($library['attributes'])) {
                foreach ($library['attributes'] as $attribute => $attribute_value) {
                  foreach ($tag_elements as $dom_element) {
                    if ($dom_element->hasAttributes() && $dom_element->attributes->getNamedItem($attribute)) {
                      $dom_element_attributes = explode(' ', $dom_element->attributes->getNamedItem($attribute)->nodeValue);
                      if (in_array($attribute_value, $dom_element_attributes)) {
                        $attachments['library'][] = $library['library'];
                      }
                    }
                  }
                }
              }
              else {
                $attachments['library'][] = $library['library'];
              }
            }
          }
        }
        $response->setAttachments($attachments);
      }
    }
    return $response;
  }

}
